/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_system.hpp
 *
 * @brief [LEVEL: beta] Stream_Test_System Class
 *
 * $Id: stream_test_system.hpp,v 1.5 2005/06/23 09:54:26 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/stream_test_system.t.cpp
 *
 */
#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__STREAM_TEST_SYSTEM_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__STREAM_TEST_SYSTEM_HPP__INCLUDE_GUARD


// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used as base class
#include <diagnostics/unittest/test_system/test_system.hpp>

// used in the interface by reference
#include <string>

// used in the interface by reference
#include <iostream>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

// used in the interface by reference
class Stream_Test_System_Implementation;

/**
 * @class Stream_Test_System diagnostics/unittest/test_system/stream_test_system.hpp
 *
 * @brief Uses cin/cout for communication with the user. 
 *
 * If started without arguments, it enters a command-loop, accepting
 * threee commands: 
 * @arg run 'mask' 'level' selects all Test_Cases which match the
 * 'mask' and the 'level' and executes them.
 * @arg list 'mask' 'level' selects all Test_Cases which match the 'mask'
 * and 'level' and lists them.  
 * @arg quit terminates the session.
 *
 * If started with arguments, they have to match the format run 'mask'
 * 'level' or list 'mask' 'level' with the same meaning as in an
 * interactive session.
 *
 * By default, as @ref Test_Data_Source the file test.dat and as log
 * file result.log is taken.
 *
 * @nosubgrouping
 */
class Stream_Test_System :
    public Test_System
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief Stream_Test_System is a @ref Test_System which is
     * intended for user-interactive testing. It takes the filename
     * for a default test database (@a default_test_data_file_name),
     * the current @a build_level, a number of arguments in terms of a
     * @a argc, @a argv pair, and three streams for user-interaction:
     * @a in for user input and @a out to prompt the user. In addition
     * a log file with name (@a default_test_data_file_name) is opened
     * for permanent logging.
     *
     * @pre @a build_level < @ref LEVEL_TEST
     * @pre @a argc / @a argv contain either nothing, or a
     * run-command, or a list-command: "<opions> run 'mask' 'level'"
     * or "<options> list 'mask' 'level'" with
     * mask_adheres_convention('mask') and
     * 0<='level'<=LEVEL_TES=3. <options> equals either "", "-i",
     * "-a", or "-r".  "-i" for interactive mode, "-a" for accepting
     * mode, "-r" for rejecting mode. Default is rejecting mode.
     * @pre TEST_DATA_FILE_NAME is a test data file in proper format, 
     * or does not exist but can be created
     * @pre RESULT_LOG_FILE_NAME can be opened for appending
     *
     * @note current TEST_DATA_FILE_NAME = @a default_test_data_file_name
     *       and RESULT_LOG_FILE_NAME = @a default_result_log_file_name
     *       but it is planed to make them subject to @a argc / @a argv 
     *       options.
     *
     * @throw Test_System_Exception
     */
    Stream_Test_System(Level_t const build_level,
					   int const argc,
					   char const * const * argv,
					   ::std::string const & default_test_data_file_name,
					   ::std::string const & default_result_log_file_name,
					   ::std::istream & in,
					   ::std::ostream & out);
    
    /**
     * @brief Save the current test data into @a default_test_data_file_name.
     * 
	 * If this is not possible, @ref DIAGNOSTICS_PANIC_LOG is used for
	 * reporting and the data is dropped silently.
     *
     * @throw never
     */
    virtual ~Stream_Test_System();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:    
    virtual void add(Test_Suite * const test_suite);
    virtual void add(Test_Case * const test_case);

    /**
     * @brief The main loop of the @ref Stream_Test_System is entered:
     * If argc/argv of the Constructor contained a "run 'mask'
     * 'level'" or a "list 'mask' 'level'" command, this command is
     * executed -- without interactions in case of missing or
     * mismatching Test_Data entries. Otherwise the user is prompted
     * for a commands.
     *
     * The commands: run and list as described above. And "interactive
     * '0/1/2'" to switch on the interactive mode, or to switch it off
     * while rejecting (1)/ accepting (2) all missing/mismatching
     * entries. In interactive mode, missing or mismatching entries
     * are resolved with user interaction. In non interactive mode,
     * the Test_Data is never modified (typically causing the
     * corresponding @ref Test_Case to fail) or always modified.
     *
     * @note Future enhancements of run: Changing the logfile and the
     * file with the test_data. Silent exeuction -- only log-output.
	 *
	 * @return if called interactively, the return value is
	 * undefined. If called non-interactively, the return value is a
	 * flag mask: TEST_SYSTEM_RESULT_FAILURE is present for test failures,
	 * TEST_SYSTEM_RESULT_INVALID is present for invalid test runs. 
     * 
     * @throw never
     */
    virtual int run();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Insulated State
     * @{
     */
private:
    Stream_Test_System_Implementation * m_impl;
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
