/*
 * Copyright (c) 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
//! Common definitions for the feature-check crate's modules.

use std::collections::HashMap;
use std::error;

/// The default option to pass to a program to obtain the list of features.
pub const DEFAULT_OPTION_NAME: &str = "--features";

/// The default prefix to look for in the lines output by the program.
pub const DEFAULT_PREFIX: &str = "Features: ";

/// The feature-check operating mode, usually "List".
#[derive(Debug)]
pub enum Mode {
    /// Obtain the list of the program's features.
    List,
    /// Obtain the value of a single feature.
    Single(String),
    /// Evaluate a "feature op version" expression.
    Simple(String),
}

/// Runtime configuration settings for
/// the [`obtain_features`][crate::obtain::obtain_features] function.
#[derive(Debug)]
pub struct Config {
    /// The option to pass to the program to query for supported features.
    pub option_name: String,
    /// The prefix to look for in the lines output by the program.
    pub prefix: String,
    /// The name or full path of the program to execute.
    pub program: String,
    /// The feature-check tool's operating mode.
    pub mode: Mode,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            option_name: DEFAULT_OPTION_NAME.to_string(),
            prefix: DEFAULT_PREFIX.to_string(),
            program: "".to_string(),
            mode: Mode::List,
        }
    }
}

/// The result of querying a program for its supported features.
#[derive(Debug)]
pub enum Obtained {
    /// The program could not be executed at all, or its output could
    /// not be parsed as a reasonable string.
    Failed(Box<dyn error::Error>),
    /// The program does not support being queried for features.
    NotSupported,
    /// The program's supported features were successfully parsed.
    Features(HashMap<String, String>),
}
