/*
 * @(#) video_device.c - Abstraction layer for a video capture device
 *
 * (C) 2001, Nick Andrew <nick@nick-andrew.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "video_device.h"

#include <assert.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <stdio.h>
#include <stdlib.h>

// Create a new instance of video_device_t

video_device_t	*vd_setup(int width, int height, int depth, int dev) {
	video_device_t *vd = (video_device_t *)malloc(sizeof(video_device_t));
	if (!vd) return vd;

	vd->width = width;
	vd->height = height;
	vd->depth = depth;
	vd->buffer_size = width * height * depth;
	vd->dev = dev;
	vd->use_mmap = 0;
	vd->capturing = 0;
	vd->debug = 0;
	vd->debug_fp = 0;

	return vd;
}

// Decide whether to capture frames using read() or mmap() and ioctl
// and if so, begin capturing frames

int	vd_setup_capture_mode(video_device_t *vd) {

	assert(vd != (video_device_t *)0);

	if (!vd_get_capabilities(vd)) {
		return 0;
	}

	// See if we can use mmap (to avoid copying data around)
	// VIDIOCGMBUF tells us how many frames it is going to buffer
	// for us, and we have to use them all!!! ???

	if (vd_ioctl(vd, VIDIOCGMBUF, &vd->vmbuf)) {
		if (vd->use_mmap) {
			fprintf(stderr, "Reversing earlier decision to use mmap\n");
			vd->use_mmap = 0;
		}

		// Issue VIDIOCGWIN to tell the driver what geometry we
		// expect from read()

		if (!vd_ioctl(vd, VIDIOCGWIN, &vd->vwin)) {
			vd->vwin.width = vd->width;
			vd->vwin.height = vd->height;
			if (vd_ioctl(vd, VIDIOCSWIN, &vd->vwin)) {
				perror("VIDIOCSWIN");
				return 0;
			}

			// Now get the geometry which the driver allowed us
			if (vd_ioctl(vd, VIDIOCSWIN, &vd->vwin)) {
				perror("VIDIOCGWIN 2");
				return 0;
			}

			if (vd->vwin.width != vd->width || vd->vwin.height != vd->height) {

				if (vd->vwin.width != vd->width) {
					fprintf(stderr, "Warning: driver changed width to %d\n", vd->vwin.width);
					vd->width = vd->vwin.width;
				}

				if (vd->vwin.height != vd->height) {
					fprintf(stderr, "Warning: driver changed height to %d\n", vd->vwin.height);
					vd->height = vd->vwin.height;
				}

				vd->buffer_size = vd->height * vd->width;
			}
		}

		vd->frame_buffer = malloc(vd->buffer_size);
		if (!vd->frame_buffer) {
			perror("malloc vd->frame_buffer");
			return 0;
		}

		return 1;
	}

	fprintf(stderr, "VIDIOCGMBUF return: size = %d\n", vd->vmbuf.size);
	fprintf(stderr, "frames: %d\n", vd->vmbuf.frames);
	fprintf(stderr, "\n");

	// mmap is okay!
	if (!vd->use_mmap) {
		fprintf(stderr, "Reversing earlier decision to NOT use mmap\n");
		vd->use_mmap = 1;
	}

	vd->frame_buffer = mmap(0, vd->vmbuf.size, PROT_READ|PROT_WRITE, MAP_SHARED, vd->dev, 0);
	if (vd->frame_buffer == (unsigned char *) -1) {
		perror("mmap vd->frame_buffer");
		return 0;
	}

	vd->vmmap.format = VIDEO_PALETTE_RGB24;		// KLUDGE ...
	vd->vmmap.frame = 0;				// Start at frame 0
	vd->vmmap.width = vd->width;
	vd->vmmap.height = vd->height;

	fprintf(stderr, "Set Format: %08x\n", vd->vmmap.format);
	fprintf(stderr, "Set Frame: %d\n", vd->vmmap.frame);
	fprintf(stderr, "Set Width: %d\n", vd->vmmap.width);
	fprintf(stderr, "Set Height: %d\n", vd->vmmap.height);

	return 1;
}

// Destroy the instance of video_device_t (close device, free memory etc)
void	vd_close(video_device_t *vd) {
	assert(vd != (video_device_t *)0);

	if (vd->frame_buffer) {
		if (vd->use_mmap) {
			munmap(vd->frame_buffer, vd->vmbuf.size);
		} else {
			free(vd->frame_buffer);
		}
	}

	close(vd->dev);

	free(vd);
}

// Get the device capabilities (internal function)
int	vd_get_capabilities(video_device_t *vd) {
	FILE	*debug_fp;

	assert(vd != (video_device_t *)0);

	debug_fp = vd->debug_fp;

	if (vd_ioctl(vd, VIDIOCGCAP, &vd->vcap)) {
		perror("VIDIOGCAP failed");
		return 0;
	}

	if (!(vd->vcap.type & VID_TYPE_CAPTURE)) {
		vd->use_mmap = 0;
	} else {
		vd->use_mmap = 1;
	}

	if (vd->debug) {

		fprintf(debug_fp, "Video device canonical Name: %s\n", vd->vcap.name);
		fprintf(debug_fp, "Type Flags: %08x\n", vd->vcap.type);
		fprintf(debug_fp, "Number of channels: %d\n", vd->vcap.channels);
		fprintf(debug_fp, "Number of audio devices: %d\n", vd->vcap.audios);
		fprintf(debug_fp, "Maximum width x height: %dx%d\n", vd->vcap.maxwidth, vd->vcap.maxheight);
		fprintf(debug_fp, "Minimum width x height: %dx%d\n", vd->vcap.minwidth, vd->vcap.minheight);
	}

	if (vd->width > vd->vcap.maxwidth || vd->width < vd->vcap.minwidth) {
		fprintf(debug_fp, "Impossible width %d\n", vd->width);
		return 0;
	}

	if (vd->height > vd->vcap.maxheight || vd->height < vd->vcap.minheight) {
		fprintf(debug_fp, "Impossible height %d\n", vd->height);
		return 0;
	}

	return 1;
}

/*
 * Each video4linux video or audio device captures from one or more
 * source channels.
 *
 * Set the channel we will use, and the "norm" for capturing that channel.
 * (input == -1) => no change to the channel ?
 *
 * Return 0 if error
 */

// Set the current video source for this device
int	vd_setup_video_source(video_device_t *vd, int input, int norm) {

	assert(vd != (video_device_t *)0);

	vd->vchan.channel = input;	// Query desired channel

	if (vd_ioctl(vd, VIDIOCGCHAN, &vd->vchan)) {
		perror("VIDIOCGCHAN");
		return 0;
	}

	// Now set the channel and the norm for this channel

	vd->vchan.norm = norm;

	if (vd_ioctl(vd, VIDIOCSCHAN, &vd->vchan)) {
		perror("VIDIOCSCHAN");
		return 0;
	}

	// KLUDGE ... the API leaves colour settings and tuning undefined
	// after a channel change
	return 1;
}

// Get an image
unsigned char *	vd_get_image(video_device_t *vd) {
	int	len;

	assert(vd != (video_device_t *)0);

	if (vd->use_mmap) {

		if (!vd->capturing) {
			int i;

			// Queue requests to capture successive frames
			for (i = 0; i < vd->vmbuf.frames; ++i) {
				vd->vmmap.frame = i;
				if (vd_ioctl(vd, VIDIOCMCAPTURE, &vd->vmmap)) {
					perror("VIDIOCMCAPTURE");
					return 0;
				}
			}

			// And start reading from zero
			vd->vmmap.frame = 0;

			vd->capturing = 1;
		}


		// VIDIOCSYNC causes the driver to block until the specified
		// frame is completely received

		if (ioctl(vd->dev, VIDIOCSYNC, &vd->vmmap.frame)) {
			perror("VIDIOCSYNC");
			return 0;
		}

		// Return the buffer, cause it should contain an image
		return vd->frame_buffer + vd->vmbuf.offsets[vd->vmmap.frame];
	}

	// Otherwise, we have to read the right number of bytes

	len = read(vd->dev, vd->frame_buffer, vd->buffer_size);
	if (len <= 0) {
		return 0;
	}

	if (len != vd->buffer_size) {
		fprintf(stderr, "Expected to read %d bytes, actually read %d\n", vd->buffer_size, len);
		return 0;
	}

	return vd->frame_buffer;
}

// Tell the device driver we are done with an image, and so capture further
// frames into that
int	vd_image_done(video_device_t *vd) {

	assert(vd != (video_device_t *)0);

	if (vd->use_mmap) {
		// vd->vmmap.frame contains the index of the recently-used buffer
		// So tell the driver to reuse this one for the next frame

		if (ioctl(vd->dev, VIDIOCMCAPTURE, &vd->vmmap)) {
			perror("VIDIOCMCAPTURE");
			return 0;
		}

		// Now cycle the frame number, so we sync the next frame
		if (++vd->vmmap.frame >= vd->vmbuf.frames) {
			vd->vmmap.frame = 0;
		}
	}

	return 1;
}

// Turn debugging (ioctl logging) on or off
void	vd_debug(video_device_t *vd, int flag, FILE *fp) {

	if (flag < 0 || flag > 1) return;

	vd->debug = flag;
	vd->debug_fp = fp;
}

//
int	vd_ioctl(video_device_t *vd, int cmd, void *arg) {
	char	*ioctl_name;
	char	ioctl_name_unknown[11];
	int	rc;
	FILE	*debug_fp;
	union	{
		struct video_capability *vcap;
		struct video_channel *vchan;
		struct video_mbuf *vmbuf;
		struct video_mmap *vmmap;
		struct video_window *vwin;
		void *vp;
	} ptr;

	// Just do the ioctl if not debugging
	if (!vd->debug) {
		return ioctl(vd->dev, cmd, arg);
	}

	debug_fp = vd->debug_fp;
	ptr.vp = arg;

	// Convert the ioctl into a name

	switch(cmd) {
		case VIDIOCGMBUF:
			ioctl_name = "VIDIOCGMBUF";
			break;
		case VIDIOCGWIN:
			ioctl_name = "VIDIOCGWIN";
			break;
		case VIDIOCSWIN:
			ioctl_name = "VIDIOCSWIN";
			break;
		case VIDIOCGCAP:
			ioctl_name = "VIDIOCGCAP";
			break;
		case VIDIOCGCHAN:
			ioctl_name = "VIDIOCGCHAN";
			fprintf(debug_fp, "VIDIOCGCHAN: Channel number %d\n", ptr.vchan->channel);
			break;
		case VIDIOCSCHAN:
			ioctl_name = "VIDIOCSCHAN";
			break;
		case VIDIOCMCAPTURE:
			ioctl_name = "VIDIOCMCAPTURE";
			break;
		case VIDIOCSYNC:
			ioctl_name = "VIDIOCSYNC";
			break;
		default:
			sprintf(ioctl_name_unknown, "0x%08lx", cmd);
			ioctl_name = ioctl_name_unknown;
	}


	fprintf(vd->debug_fp, "Issue ioctl: %s ", ioctl_name);

	rc = ioctl(vd->dev, cmd, arg);
	fprintf(vd->debug_fp, "(rc %d)\n", rc);

	if (rc == -1) {
		return rc;
	}

	switch(cmd) {
		case VIDIOCGMBUF:
			break;
		case VIDIOCGWIN:
			break;
		case VIDIOCSWIN:
			break;
		case VIDIOCGCAP:
			fprintf(debug_fp, "VIDIOCGCAP: %s\n", ptr.vcap->name);
			break;
		case VIDIOCGCHAN:
			fprintf(debug_fp, "Channel name: %s\n", ptr.vchan->name);
			fprintf(debug_fp, "Number of tuners: %d\n", ptr.vchan->tuners);
			fprintf(debug_fp, "Flags: %08x\n", ptr.vchan->flags);
			fprintf(debug_fp, "Type: %08x\n", ptr.vchan->type);
			fprintf(debug_fp, "Norm: %08x\n", ptr.vchan->norm);
			break;
		case VIDIOCSCHAN:
			break;
		case VIDIOCMCAPTURE:
			break;
		case VIDIOCSYNC:
			break;
	}


	return rc;
}

