/*
 *  $Id: grains.c 28952 2025-12-05 18:34:48Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>

#include "libgwyddion/macros.h"
#include "libgwyddion/linestats.h"
#include "libgwyddion/filters.h"
#include "libgwyddion/arithmetic.h"
#include "libgwyddion/stats.h"
#include "libgwyddion/grains.h"

#include "libgwyddion/int-list.h"
#include "libgwyddion/omp.h"
#include "libgwyddion/internal.h"

static gint*   gwy_field_fill_grain    (GwyField *field,
                                        gint col,
                                        gint row,
                                        gint *nindices);
static gint    gwy_field_fill_one_grain(gint xres,
                                        gint yres,
                                        const gint *data,
                                        gint col,
                                        gint row,
                                        gint *visited,
                                        gint grain_no,
                                        IntList *listv,
                                        IntList *listh);

/**
 * gwy_field_grains_mark_height:
 * @field: Data to be used for marking.
 * @grain_field: Data field to store the resulting mask to.
 * @threshval: Relative height threshold, in percents.
 * @below: If %TRUE, data below threshold are marked, otherwise data above threshold are marked.
 *
 * Marks data that are above/below height threshold.
 **/
void
gwy_field_grains_mark_height(GwyField *field,
                             GwyField *grain_field,
                             gdouble threshval,
                             gboolean below)
{
    gdouble min, max;

    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    gwy_field_copy_data(field, grain_field);
    gwy_field_get_min_max(grain_field, &min, &max);
    if (below)
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 1, 0);
    else
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 0, 1);

    gwy_field_invalidate(grain_field);
}

/**
 * gwy_field_grains_mark_curvature:
 * @field: Data to be used for marking.
 * @grain_field: Data field to store the resulting mask to.
 * @threshval: Relative curvature threshold, in percents.
 * @below: If %TRUE, data below threshold are marked, otherwise data above threshold are marked.
 *
 * Marks data that are above/below curvature threshold.
 **/
void
gwy_field_grains_mark_curvature(GwyField *field,
                                GwyField *grain_field,
                                gdouble threshval,
                                gboolean below)
{
    gdouble min, max;

    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    gwy_field_copy_data(field, grain_field);
    gwy_field_filter_laplacian(grain_field);

    gwy_field_get_min_max(grain_field, &min, &max);
    if (below)
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 1, 0);
    else
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 0, 1);

    gwy_field_invalidate(grain_field);
}

/**
 * gwy_field_grains_mark_slope:
 * @field: Data to be used for marking.
 * @grain_field: Data field to store the resulting mask to.
 * @threshval: Relative slope threshold, in percents.
 * @below: If %TRUE, data below threshold are marked, otherwise data above threshold are marked.
 *
 * Marks data that are above/below slope threshold.
 **/
void
gwy_field_grains_mark_slope(GwyField *field,
                            GwyField *grain_field,
                            gdouble threshval,
                            gboolean below)
{
    GwyField *masky;
    const gdouble *mdata;
    gdouble *gdata;
    gdouble min, max;

    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    gint xres = field->xres, yres = field->yres;
    gsize n = (gsize)xres * (gsize)yres;

    masky = gwy_field_copy(field);
    gwy_field_copy_data(field, grain_field);
    gwy_field_filter_sobel(grain_field, GWY_ORIENTATION_HORIZONTAL);
    gwy_field_filter_sobel(masky, GWY_ORIENTATION_VERTICAL);

    gdata = grain_field->priv->data;
    mdata = masky->priv->data;
#ifdef _OPENMP
#pragma omp parallel for if(gwy_threads_are_enabled()) default(none) \
            shared(gdata,mdata,n)
#endif
    for (gsize i = 0; i < n; i++)
        gdata[i] = sqrt(gdata[i]*gdata[i] + mdata[i]*mdata[i]);

    gwy_field_get_min_max(grain_field, &min, &max);
    if (below)
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 1, 0);
    else
        gwy_field_threshold(grain_field, min + threshval*(max - min)/100.0, 0, 1);

    g_object_unref(masky);
    gwy_field_invalidate(grain_field);
}

/**
 * gwy_field_grains_remove_grain:
 * @grain_field: Field of marked grains (mask).
 * @col: Column inside a grain.
 * @row: Row inside a grain.
 *
 * Removes one grain at given position.
 *
 * Returns: %TRUE if a grain was actually removed, i.e. (@col,@row) was inside a grain.
 **/
gboolean
gwy_field_grains_remove_grain(GwyField *grain_field,
                              gint col,
                              gint row)
{
    gint *points;
    gint npoints = 0;

    g_return_val_if_fail(GWY_IS_FIELD(grain_field), FALSE);
    g_return_val_if_fail(col >= 0 && col < grain_field->xres, FALSE);
    g_return_val_if_fail(row >= 0 && row < grain_field->yres, FALSE);

    gdouble *d = grain_field->priv->data;
    if (!d[grain_field->xres*row + col])
        return FALSE;

    points = gwy_field_fill_grain(grain_field, col, row, &npoints);
    while (npoints) {
        npoints--;
        d[points[npoints]] = 0.0;
    }
    g_free(points);
    gwy_field_invalidate(grain_field);

    return TRUE;
}

/**
 * gwy_field_grains_extract_grain:
 * @grain_field: Field of marked grains (mask).
 * @col: Column inside a grain.
 * @row: Row inside a grain.
 *
 * Removes all grains except that one at given position.
 *
 * If there is no grain at (@col, @row), all grains are removed.
 *
 * Returns: %TRUE if a grain remained (i.e., (@col,@row) was inside a grain).
 **/
gboolean
gwy_field_grains_extract_grain(GwyField *grain_field,
                               gint col,
                               gint row)
{
    gint *points;
    gint npoints = 0;

    g_return_val_if_fail(GWY_IS_FIELD(grain_field), FALSE);
    g_return_val_if_fail(col >= 0 && col < grain_field->xres, FALSE);
    g_return_val_if_fail(row >= 0 && row < grain_field->yres, FALSE);

    gdouble *d = grain_field->priv->data;
    if (!d[grain_field->xres*row + col]) {
        gwy_field_clear(grain_field);
        return FALSE;
    }

    points = gwy_field_fill_grain(grain_field, col, row, &npoints);
    gwy_field_clear(grain_field);
    while (npoints) {
        npoints--;
        d[points[npoints]] = 1.0;
    }
    g_free(points);
    gwy_field_invalidate(grain_field);

    return TRUE;
}

/**
 * gwy_field_grains_remove_by_number:
 * @grain_field: Field of marked grains (mask).
 * @number: Grain number was filled by gwy_field_number_grains().
 *
 * Removes grain identified by @number.
 **/
void
gwy_field_grains_remove_by_number(GwyField *grain_field,
                                  gint number)
{
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    gint xres = grain_field->xres, yres = grain_field->yres;
    gsize n = (gsize)xres * (gsize)yres;
    gdouble *data = grain_field->priv->data;

    gint *grains = g_new0(gint, n);
    gwy_field_number_grains(grain_field, grains);

    for (gsize i = 0; i < n; i++) {
        if (grains[i] == number)
            data[i] = 0;
    }

    g_free(grains);
}

/**
 * gwy_field_grains_remove_by_size:
 * @grain_field: Field of marked grains (mask).
 * @size: Grain area threshold, in square pixels.
 *
 * Removes all grains below specified area.
 **/
void
gwy_field_grains_remove_by_size(GwyField *grain_field,
                                gint size)
{
    gint i, n, ngrains;
    gdouble *data;
    gint *grain_size;
    gint *grains;

    g_return_if_fail(GWY_IS_FIELD(grain_field));

    n = grain_field->xres * grain_field->yres;
    data = grain_field->priv->data;

    grains = g_new0(gint, n);
    ngrains = gwy_field_number_grains(grain_field, grains);
    grain_size = gwy_field_get_grain_sizes(grain_field, ngrains, grains, NULL);
    /* Avoid some no-op work below. */
    grain_size[0] = size;

    /* Too trivial to parallelise. */
    /* remove grains */
    for (i = 0; i < n; i++) {
        if (grain_size[grains[i]] < size)
            data[i] = 0;
    }
    g_free(grains);

    for (i = 1; i <= ngrains; i++) {
        if (grain_size[i] < size) {
            gwy_field_invalidate(grain_field);
            break;
        }
    }
    g_free(grain_size);
}

/**
 * gwy_field_grains_remove_by_height:
 * @field: Data to be used for marking
 * @grain_field: Field of marked grains (mask)
 * @threshval: Relative height threshold, in percents.
 * @below: If %TRUE, grains below threshold are removed, otherwise grains above threshold are removed.
 *
 * Removes grains that are higher/lower than given threshold value.
 **/
void
gwy_field_grains_remove_by_height(GwyField *field,
                                  GwyField *grain_field,
                                  gdouble threshval,
                                  gboolean below)
{
    gint i, n, ngrains;
    gdouble *data;
    gboolean *grain_kill;
    gdouble min, max;
    gint *grains;

    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    n = grain_field->xres * grain_field->yres;
    data = grain_field->priv->data;

    gwy_field_get_min_max(field, &min, &max);
    threshval = min + threshval*(max - min)/100.0;

    grains = g_new0(gint, n);
    ngrains = gwy_field_number_grains(grain_field, grains);

    /* find grains to remove */
    grain_kill = g_new0(gboolean, ngrains + 1);
    if (below) {
        /* Too trivial to parallelise. */
        for (i = 0; i < n; i++) {
            if (grains[i] && data[i] < threshval)
                grain_kill[grains[i]] = TRUE;
        }
    }
    else {
        /* Too trivial to parallelise. */
        for (i = 0; i < n; i++) {
            if (grains[i] && data[i] > threshval)
                grain_kill[grains[i]] = TRUE;
        }
    }
    /* Avoid some no-op work below. */
    grain_kill[0] = FALSE;

    /* remove them */
    /* Too trivial to parallelise. */
    for (i = 0; i < n; i++) {
        if (grain_kill[grains[i]])
            data[i] = 0;
    }
    for (i = 1; i <= ngrains; i++) {
        if (grain_kill[i]) {
            gwy_field_invalidate(grain_field);
            break;
        }
    }

    g_free(grains);
    g_free(grain_kill);
}

/**
 * gwy_field_grains_remove_touching_border:
 * @grain_field: Field of marked grains (mask).
 *
 * Removes all grains that touch field borders.
 **/
void
gwy_field_grains_remove_touching_border(GwyField *grain_field)
{
    gint i, xres, yres, ngrains;
    gdouble *data;
    gint *grains;
    gboolean *touching;

    g_return_if_fail(GWY_IS_FIELD(grain_field));

    xres = grain_field->xres;
    yres = grain_field->yres;
    data = grain_field->priv->data;

    grains = g_new0(gint, xres*yres);
    ngrains = gwy_field_number_grains(grain_field, grains);

    /* Remember grains that touch any border. */
    touching = g_new0(gboolean, ngrains + 1);
    for (i = 0; i < xres; i++)
        touching[grains[i]] = TRUE;
    for (i = 1; i < yres-1; i++) {
        touching[grains[i*xres]] = TRUE;
        touching[grains[i*xres + xres-1]] = TRUE;
    }
    for (i = 0; i < xres; i++)
        touching[grains[(yres-1)*xres + i]] = TRUE;
    /* Avoid some no-op work below. */
    touching[0] = FALSE;

    /* Remove grains. */
    for (i = 0; i < xres*yres; i++) {
        if (touching[grains[i]])
            data[i] = 0;
    }
    for (i = 1; i <= ngrains; i++) {
        if (touching[i]) {
            gwy_field_invalidate(grain_field);
            break;
        }
    }

    g_free(grains);
    g_free(touching);
}

/**
 * gwy_field_grains_add:
 * @grain_field: Field of marked grains (mask).
 * @add_field: Field of marked grains (mask) to be added.
 *
 * Adds @add_field grains to @grain_field.
 *
 * Note: This function is equivalent to
 * |[
 * gwy_field_max_of_fields(grain_field, grain_field, add_field);
 * ]|
 **/
void
gwy_field_grains_add(GwyField *grain_field, GwyField *add_field)
{
    gwy_field_max_of_fields(grain_field, grain_field, add_field);
}

/**
 * gwy_field_grains_intersect:
 * @grain_field: Field of marked grains (mask).
 * @intersect_field: Field of marked grains (mask).
 *
 * Performs intersection betweet two grain fields, result is stored in @grain_field.
 *
 * Note: This function is equivalent to
 * |[
 * gwy_field_min_of_fields(grain_field, grain_field, intersect_field);
 * ]|
 **/
void
gwy_field_grains_intersect(GwyField *grain_field,
                           GwyField *intersect_field)
{
    gwy_field_min_of_fields(grain_field, grain_field, intersect_field);
}

/**
 * gwy_field_grains_invert:
 * @grain_field: Data field (mask) of marked grains.
 *
 * Inverts a data field representing a mask.
 *
 * All non-positive values are transformed to 1.0.  All positive values are transformed to 0.0.
 **/
void
gwy_field_grains_invert(GwyField *grain_field)
{
    g_return_if_fail(GWY_IS_FIELD(grain_field));

    gint xres = grain_field->xres, yres = grain_field->yres;
    gsize n = (gsize)xres * (gsize)yres;
    gdouble *d = grain_field->priv->data;
    for (gsize k = 0; k < n; k++)
        d[k] = !(d[k] > 0.0);

    gwy_field_invalidate(grain_field);
}

/**
 * gwy_field_grains_autocrop:
 * @mask_field: Data field representing a mask.
 * @symmetrically: %TRUE to remove borders symmetrically, i.e the same number of pixels from left and right, and also
 *                 top and bottom. %FALSE to remove as many empty rows and columns as possible.
 * @left: (out) (optional): Location to store how many column were removed from the left, or %NULL.
 * @right: (out) (optional): Location to store how many column were removed from the right, or %NULL.
 * @up: (out) (optional): Location to store how many row were removed from the top, or %NULL.
 * @down: (out) (optional): Location to store how many row were removed from the bottom, or %NULL.
 *
 * Removes empty border rows and columns from a data field representing a mask.
 *
 * If there are border rows and columns filled completely with non-positive values the size of the data field is
 * reduced, removing these rows.  The parameter @symmetrically controls whether the size reduction is maximum possible
 * or symmetrical.
 *
 * When there is no positive value in the field the field size is reduced to the smallest possible.  This means 1x1
 * for @symmetrical being %FALSE and even original dimensions to 2 for @symmetrical being %TRUE.
 *
 * Returns: %TRUE if the field size was reduced at all.  Detailed information about the reduction can be obtained from
 *          @left, @right, @up and @down.
 **/
gboolean
gwy_field_grains_autocrop(GwyField *mask_field,
                          gboolean symmetrically,
                          guint *left,
                          guint *right,
                          guint *up,
                          guint *down)
{
    gint xres, yres, i, j, firstcol, firstrow, lastcol, lastrow;
    const gdouble *d;

    g_return_val_if_fail(GWY_IS_FIELD(mask_field), FALSE);
    xres = mask_field->xres;
    yres = mask_field->yres;
    firstcol = xres;
    firstrow = yres;
    lastcol = lastrow = -1;
    d = mask_field->priv->data;
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++, d++) {
            if (*d > 0.0) {
                if (G_UNLIKELY(i < firstrow))
                    firstrow = i;
                if (G_UNLIKELY(j < firstcol))
                    firstcol = j;
                if (G_UNLIKELY(i > lastrow))
                    lastrow = i;
                if (G_UNLIKELY(j > lastcol))
                    lastcol = j;
            }
        }
    }
    gwy_debug("first (%d,%d) last (%d,%d)",
              firstcol, firstrow, lastcol, lastrow);
    if (firstcol > lastcol) {
        g_assert(firstrow > lastrow);
        /* Anticipate the reduction to 2 for even-sized dimensions. */
        lastcol = (xres - 1)/2;
        firstcol = xres - lastcol;
        lastrow = (yres - 1)/2;
        firstrow = yres - lastrow;
    }
    if (symmetrically) {
        firstcol = MIN(firstcol, xres-1 - lastcol);
        lastcol = xres-1 - firstcol;
        firstrow = MIN(firstrow, yres-1 - lastrow);
        lastrow = yres-1 - firstrow;
    }
    lastcol++;
    lastrow++;

    if (left)
        *left = firstcol;
    if (right)
        *right = xres - lastcol;
    if (up)
        *up = firstrow;
    if (down)
        *down = yres - lastrow;

    gwy_debug("%dx%d at (%d,%d) of %dx%d",
              lastcol-firstcol, lastrow-firstrow, firstcol, firstrow, xres, yres);
    if (firstcol == 0 && firstrow == 0 && lastcol == xres && lastrow == yres)
        return FALSE;

    gwy_field_crop(mask_field, firstcol, firstrow, lastcol-firstcol, lastrow-firstrow);
    return TRUE;
}

static inline gint
finalise_grain_numbering(gssize *m, gint max_id,
                         gint *grains, gint n)
{
    gint *mm;
    gint i, id;

    /* Resolve remianing grain number links in map */
    for (i = 1; i <= max_id; i++)
        m[i] = m[m[i]];

    /* Compactify grain numbers */
    mm = g_new0(gint, max_id + 1);
    id = 0;
    for (i = 1; i <= max_id; i++) {
        if (!mm[m[i]]) {
            id++;
            mm[m[i]] = id;
        }
        m[i] = mm[m[i]];
    }
    g_free(mm);

    /* Renumber grains (we make use of the fact m[0] = 0) */
    for (i = 0; i < n; i++)
        grains[i] = m[grains[i]];

    return id;
}

/* Given an image of integers, with zeros corresponding to non-grains and any non-zero values to grains, number
 * grains.  In other words, the non-zero values become grain numbers while zeros are untouched. */
static gint
renumber_grains(gint *grains, gint xres, gint yres, IntList *m)
{
    gint i, j, k, grain_id, max_id, id;
    gboolean must_free = FALSE;

    g_return_val_if_fail(grains, 0);

    if (!m) {
        m = int_list_new(xres + yres);
        must_free = TRUE;
    }
    else
        m->len = 0;

    int_list_add(m, 0);

    /* Number grains with simple unidirectional grain number propagation, updating map m for later full grain join */
    /* OpenMP: This seems too simple and fast to parallelise reasonably. */
    max_id = 0;
    grain_id = 0;
    k = 0;
    for (i = 0; i < yres; i++) {
        grain_id = 0;
        for (j = 0; j < xres; j++, k++) {
            if (grains[k]) {
                /* Grain number is kept from left neighbour unless it does not exist (a new number is assigned) or
                 * a join with top neighbour occurs (m is updated) */
                if (i > 0 && (id = grains[k-xres])) {
                    if (!grain_id)
                        grain_id = id;
                    else if (id != grain_id) {
                        resolve_grain_map(m->data, id, grain_id);
                        grain_id = m->data[id];
                    }
                }
                if (!grain_id) {
                    grain_id = ++max_id;
                    int_list_add(m, grain_id);
                }
                grains[k] = grain_id;
            }
            else
                grain_id = 0;
        }
    }

    max_id = finalise_grain_numbering(m->data, max_id, grains, xres*yres);
    if (must_free)
        int_list_free(m);

    return max_id;
}

/**
 * gwy_field_number_grains:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.
 * @grains: Zero-filled array of integers of equal size to @mask_field to put grain numbers to.  Empty space will be
 *          left 0, pixels inside a grain will be set to grain number.  Grains are numbered sequentially 1, 2, 3, ...
 *
 * Numbers grains in a mask data field.
 *
 * Returns: The number of last grain (note they are numbered from 1).
 **/
/* Since 2.53 the caller does not need to pre-fill grains[] with zeros, but do not advertise it much to preserve
 * compatibility... */
gint
gwy_field_number_grains(GwyField *mask_field,
                        gint *grains)
{
    g_return_val_if_fail(GWY_IS_FIELD(mask_field), 0);
    g_return_val_if_fail(grains, 0);

    gint xres = mask_field->xres, yres = mask_field->yres;
    gsize n = (gsize)xres * (gsize)yres;
    const gdouble *data = mask_field->priv->data;
    /* Too trivial to parallelise. */
    for (gsize i = 0; i < n; i++)
        grains[i] = (data[i] > 0.0);

    return renumber_grains(grains, xres, yres, NULL);
}

/**
 * gwy_field_number_grains_periodic:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.
 * @grains: Zero-filled array of integers of equal size to @mask_field to put grain numbers to.  Empty space will be
 *          left 0, pixels inside a grain will be set to grain number.  Grains are numbered sequentially 1, 2, 3, ...
 *
 * Numbers grains in a periodic mask data field.
 *
 * This function differs from gwy_field_number_grains() by the assumption of periodicity, i.e. grains can touch
 * across the opposite field edges.
 *
 * Note that some grain operations assume that grains are contiguous within the image and do not work with periodic
 * grains.
 *
 * You can use gwy_field_get_grain_sizes() and there is gwy_field_get_grain_bounding_boxes_periodic() for
 * bouding boxes of periodic grains.  As for grain quantities, simple quantities that do not depend on the shape
 * (mean, median, etc.) are evaluated correctly even for periodic grains.  You cannot evaluate quantities depending on
 * grain boundaries though.
 *
 * Returns: The number of last grain (note they are numbered from 1).
 **/
gint
gwy_field_number_grains_periodic(GwyField *mask_field,
                                 gint *grains)
{
    gint xres, yres, i, j, ngrains;
    gboolean merged_anything = FALSE;
    gssize *m;

    ngrains = gwy_field_number_grains(mask_field, grains);
    if (ngrains < 2)
        return ngrains;

    xres = mask_field->xres;
    yres = mask_field->yres;

    m = g_new0(gssize, ngrains+1);
    for (j = 0; j <= ngrains; j++)
        m[j] = j;

    for (i = 0; i < xres; i++) {
        gint gno1 = grains[i], gno2 = grains[i + (yres-1)*xres];
        if (gno1 && gno2 && gno1 != gno2) {
            resolve_grain_map(m, gno1, gno2);
            merged_anything = TRUE;
        }
    }

    for (i = 0; i < yres; i++) {
        gint gno1 = grains[i*xres], gno2 = grains[i*xres + xres-1];
        if (gno1 && gno2 && gno1 != gno2) {
            resolve_grain_map(m, gno1, gno2);
            merged_anything = TRUE;
        }
    }

    if (merged_anything)
        ngrains = finalise_grain_numbering(m, ngrains, grains, xres*yres);

    g_free(m);

    return ngrains;
}

static inline void
init_bbox_ranges(gint *bboxes, gint ngrains)
{
    gint i;

    for (i = 1; i <= ngrains; i++) {
        bboxes[4*i] = bboxes[4*i + 1] = G_MAXINT;
        bboxes[4*i + 2] = bboxes[4*i + 3] = -1;
    }
}

static inline void
extend_bbox_range(gint *bboxes, gint id, gint j, gint i)
{
    id *= 4;
    if (j < bboxes[id])
        bboxes[id] = j;
    if (i < bboxes[id + 1])
        bboxes[id + 1] = i;
    if (j > bboxes[id + 2])
        bboxes[id + 2] = j;
    if (i > bboxes[id + 3])
        bboxes[id + 3] = i;
}

static inline void
transform_range_to_bbox(gint *bbox)
{
    bbox[2] = bbox[2] + 1 - bbox[0];
    bbox[3] = bbox[3] + 1 - bbox[1];
}

/**
 * gwy_field_get_grain_bounding_boxes:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.  However its contents is
 *              ignored as all grain information is taken from @grains (its dimensions determine the dimensions of
 *              @grains).
 * @ngrains: The number of grains as returned by gwy_field_number_grains().
 * @grains: Grain numbers filled with gwy_field_number_grains().
 * @bboxes: Array of size at least 4*(@ngrains+1) to fill with grain bounding boxes.  It can be %NULL to allocate
 *          a new array.
 *
 * Find bounding boxes of all grains.
 *
 * As usual the zeroth element of @bboxes does not correspond to any grain; grain numbers start from 1. The bounding
 * boxes are stored as quadruples of indices: (column, row, width, height).
 *
 * Returns: Either @bboxes (if it was not %NULL), or a newly allocated array of size 4(@ngrains + 1).
 **/
gint*
gwy_field_get_grain_bounding_boxes(GwyField *mask_field,
                                   gint ngrains,
                                   const gint *grains,
                                   gint *bboxes)
{
    gint xres, yres, i, j, id;
    const gint *grow;

    g_return_val_if_fail(GWY_IS_FIELD(mask_field), NULL);
    g_return_val_if_fail(grains, NULL);

    xres = mask_field->xres;
    yres = mask_field->yres;
    if (!bboxes)
        bboxes = g_new(gint, 4*(ngrains + 1));

    init_bbox_ranges(bboxes, ngrains);
    for (i = 0; i < yres; i++) {
        grow = grains + i*xres;
        for (j = 0; j < xres; j++) {
            if ((id = grow[j]))
                extend_bbox_range(bboxes, id, j, i);
        }
    }
    for (i = 1; i <= ngrains; i++)
        transform_range_to_bbox(bboxes + 4*i);

    return bboxes;
}

/* Find bounding box from projection. */
static void
make_bbox_from_projection(const gboolean *projection, gint len,
                          gint *pos, gint *size)
{
    gint bl, br;

    if (projection[0] && projection[len-1]) {
        /* At most one gap in the middle, possibly entire width filled. */
        for (bl = 0; bl < len && projection[bl]; bl++)
            ;
        /* If all projection pixels are covered no smaller bbox is possible. */
        if (bl == len) {
            *pos = 0;
            *size = len;
        }
        else {
            /* There must be a gap. */
            for (br = len-1; projection[br]; br--)
                ;
            *pos = br+1;
            *size = bl + len-1-br;
        }
    }
    else {
        /* One contiguous segment, not covering the entire line.  So neither cycle needs an explicit length
         * stop-condition. */
        for (bl = 0; !projection[bl]; bl++)
            ;
        for (br = len-1; !projection[br]; br--)
            ;
        *pos = bl;
        *size = br+1 - bl;
    }
}

/**
 * gwy_field_get_grain_bounding_boxes_periodic:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.  However its contents is
 *              ignored as all grain information is taken from @grains (its dimensions determine the dimensions of
 *              @grains).
 * @ngrains: The number of grains as returned by gwy_field_number_grains_periodic().
 * @grains: Grain numbers filled with gwy_field_number_grains_periodic().
 * @bboxes: Array of size at least 4*(@ngrains+1) to fill with grain bounding boxes.  It can be %NULL to allocate
 *          a new array.
 *
 * Find bounding boxes of all grains.
 *
 * As usual zeroth element of @bboxes does not correspond to any grain; grain numbers start from 1. The bounding boxes
 * are stored as quadruples of indices: (column, row, width, height).
 *
 * The row and column always lie inside the the image.  However, width and height may specify an area which sticks
 * outside.  In this case periodicity needs to be taken into account.
 *
 * Returns: Either @bboxes (if it was not %NULL), or a newly allocated array of size 4(@ngrains + 1).
 **/
gint*
gwy_field_get_grain_bounding_boxes_periodic(GwyField *mask_field,
                                            gint ngrains,
                                            const gint *grains,
                                            gint *bboxes)
{
    gint xres, yres, i, j, ii, k, id, ntodo;
    gint *sbboxes;
    gboolean *todograins, *proj_storage;
    gboolean **xprojection, **yprojection;
    const gint *grow;

    bboxes = gwy_field_get_grain_bounding_boxes(mask_field, ngrains, grains, bboxes);
    g_return_val_if_fail(bboxes, NULL);

    xres = mask_field->xres;
    yres = mask_field->yres;

    /* Grains passing through the boundary will have full image width or height according to the normal bounding box
     * algorithm.  If there are none we can avoid the slow path. */
    todograins = g_new0(gboolean, ngrains+1);
    ntodo = 0;
    for (id = 1; id <= ngrains; id++) {
        if (bboxes[4*id + 2] == xres || bboxes[4*id + 3] == yres) {
            todograins[id] = TRUE;
            ntodo++;
        }
        else {
            gwy_debug("normal bbox[%d] %dx%d at (%d,%d)",
                      id,
                      bboxes[4*id + 2], bboxes[4*id + 3],
                      bboxes[4*id + 0], bboxes[4*id + 1]);
        }
    }
    gwy_debug("after normal bboxing %d possibly cross-border grains remaining",
              ntodo);
    if (!ntodo) {
        g_free(todograins);
        return bboxes;
    }

    /* Now try virtually shifting the image periodically by half (horizontally (0), vertically (1) and both (2)) and
     * find the bounding boxes again. This catches all grains that simply cross the boundary but are not actually too
     * large. */
    sbboxes = g_new(gint, 4*(ngrains + 1));
    for (k = 0; k < 3 && ntodo; k++) {
        init_bbox_ranges(sbboxes, ngrains);

        for (i = 0; i < yres; i++) {
            grow = grains + i*xres;
            ii = (k == 0) ? i : (i >= yres/2 ? i : i + yres);
            if (k == 1) {
                /* Horizontal lines are preserved. */
                for (j = 0; j < xres; j++) {
                    if (todograins[id = grow[j]])
                        extend_bbox_range(sbboxes, id, j, ii);
                }
            }
            else {
                /* Horizontal lines are split and shifted. */
                for (j = 0; j < xres/2; j++) {
                    if (todograins[id = grow[j]])
                        extend_bbox_range(sbboxes, id, j + xres, ii);
                }
                for (j = xres/2; j < xres; j++) {
                    if (todograins[id = grow[j]])
                        extend_bbox_range(sbboxes, id, j, ii);
                }
            }
        }

        for (id = 1; id <= ngrains; id++) {
            if (!todograins[id])
                continue;

            transform_range_to_bbox(sbboxes + 4*id);
            if (sbboxes[4*id + 2] == xres || sbboxes[4*id + 3] == yres)
                continue;

            /* We were able to find shifted finite bbox for grain @i. Furthermore if the bbox is finite now it must
             * begin inside the unshifted half -- otherwise it would lie *entirely* within the shifted half and be
             * found by the normal algorithm. */
            g_assert(sbboxes[4*id + 0] < xres);
            g_assert(sbboxes[4*id + 1] < yres);
            gwy_assign(bboxes + 4*id, sbboxes + 4*id, 4);
            gwy_debug("shifted-%d bbox[%d] %dx%d at (%d,%d)",
                      k, id, bboxes[4*id + 2], bboxes[4*id + 3], bboxes[4*id + 0], bboxes[4*id + 1]);
            todograins[id] = FALSE;
            ntodo--;
        }
    }
    g_free(sbboxes);

    gwy_debug("after shifted bboxing %d possibly cross-border grains remaining", ntodo);
    if (!ntodo) {
        g_free(todograins);
        return bboxes;
    }

    /* For any remaining grains, hopefully relatively few, find horizontal and vertical projections. */
    proj_storage = g_new0(gboolean, ntodo*(xres + yres));
    xprojection = g_new0(gboolean*, 2*(ngrains + 1));
    yprojection = xprojection + ngrains+1;
    j = 0;
    for (id = 1; id <= ngrains; id++) {
        if (!todograins[id])
            continue;

        xprojection[id] = proj_storage + j*(xres + yres);
        yprojection[id] = proj_storage + j*(xres + yres) + xres;
        j++;
    }
    g_free(todograins);

    for (i = 0; i < yres; i++) {
        grow = grains + i*xres;
        for (j = 0; j < xres; j++) {
            if (xprojection[id = grow[j]]) {
                xprojection[id][j] = TRUE;
                yprojection[id][i] = TRUE;
            }
        }
    }

    for (id = 1; id <= ngrains; id++) {
        if (!xprojection[id])
            continue;

        make_bbox_from_projection(xprojection[id], xres, bboxes + 4*id + 0, bboxes + 4*id + 2);
        make_bbox_from_projection(yprojection[id], yres, bboxes + 4*id + 1, bboxes + 4*id + 3);
    }
    g_free(xprojection);
    g_free(proj_storage);

    return bboxes;
}

static void
close_rectangles(gint *rectwidths, gint from, gint to, gint *wmax, gint *hmax)
{
    gint hm = 0, wm = 0, areamax = 0, h;

    /* Ignore the zeroth element. */
    from = MAX(from, 1);
    for (h = from; h <= to; h++) {
        if (rectwidths[h] * h > areamax) {
            areamax = rectwidths[h] * h;
            hm = h;
            wm = rectwidths[h];
        }
        rectwidths[h] = 0;
    }

    *hmax = hm;
    *wmax = wm;
}

static inline void
remember_largest_rectangle(gint *iboxes, gint gno,
                           gint j, gint i, gint wmax, gint hmax)
{
    if (!gno)
        return;

    iboxes += 4*gno;
    if (hmax*wmax > iboxes[2]*iboxes[3]) {
        iboxes[0] = j - wmax;
        iboxes[1] = i;
        iboxes[2] = wmax;
        iboxes[3] = hmax;
    }
}

/**
 * gwy_field_get_grain_inscribed_boxes:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.  However its contents is
 *              ignored as all grain information is taken from @grains (its dimensions determine the dimensions of
 *              @grains).
 * @ngrains: The number of grains as returned by gwy_field_number_grains().
 * @grains: Grain numbers filled with gwy_field_number_grains().
 * @iboxes: Array of size at least 4*(@ngrains+1) to fill with grain bounding boxes.  It can be %NULL to allocate
 *          a new array.
 *
 * Find maximum-area inscribed boxes of all grains.
 *
 * As usual zeroth element of @iboxes does not correspond to any grain; grain numbers start from 1. The bounding boxes
 * are stored as quadruples of indices: (column, row, width, height).
 *
 * Returns: Either @iboxes (if it was not %NULL), or a newly allocated array of size 4(@ngrains + 1).
 **/
gint*
gwy_field_get_grain_inscribed_boxes(GwyField *mask_field,
                                    gint ngrains,
                                    const gint *grains,
                                    gint *iboxes)
{
    gint xres, yres, i, j, k, h, hmax, wmax, gno, maxopen;
    gint *colsizes, *csrow, *rectwidths;
    const gint *grow;

    g_return_val_if_fail(GWY_IS_FIELD(mask_field), NULL);
    g_return_val_if_fail(grains, NULL);

    xres = mask_field->xres;
    yres = mask_field->yres;

    /* Pass 1.  Each value in colsizes[] will be filled with how may pixels we we can extend this one while keeping in
     * the same grain.  One means just the single pixels.
     *
     * Unlike in the standard presentation of the algorithm, we go by row for good data locality. */
    colsizes = g_new0(gint, xres*yres);
    csrow = colsizes + (yres - 1)*xres;
    grow = grains + (yres - 1)*xres;
    for (j = 0; j < xres; j++)
        csrow[j] = !!grow[j];

    for (i = yres-1; i; i--) {
        /* These rows we set, reading from the row below. */
        csrow = colsizes + (i-1)*xres;
        grow = grains + (i-1)*xres;
        for (j = 0; j < xres; j++) {
            if (grow[j]) {
                if (grow[j] == grow[xres + j])
                    csrow[j] = csrow[xres + j] + 1;
                else
                    csrow[j] = 1;
            }
        }
    }

    if (!iboxes)
        iboxes = g_new(gint, 4*(ngrains + 1));
    gwy_clear(iboxes, 4*(ngrains + 1));

    /* Pass 2.  Go from the top and close rectangles, remembering the maximum rectangle for each grain number. */
    rectwidths = g_new0(gint, yres);

    maxopen = 0;
    for (i = 0; i < yres; i++) {
        csrow = colsizes + i*xres;
        grow = grains + i*xres;
        gwy_clear(rectwidths, maxopen);
        maxopen = 0;
        gno = 0;
        for (j = 0; j < xres; j++) {
            h = csrow[j];
            /* We hit a new grain -- the algorithm works even with touching grains -- or empty space. */
            if (grow[j] != gno) {
                close_rectangles(rectwidths, 0, maxopen, &wmax, &hmax);
                remember_largest_rectangle(iboxes, gno, j, i, wmax, hmax);
            }
            else {
                /* Grain continuation.  Close rectangles higher than h (up to maxopen). */
                close_rectangles(rectwidths, h+1, maxopen, &wmax, &hmax);
                remember_largest_rectangle(iboxes, gno, j, i, wmax, hmax);
            }
            gno = grow[j];
            if (gno) {
                /* Open or extend all rectangles at most h high.  Both is realised by adding 1 because closed ones are
                 * zeros. */
                for (k = 1; k <= h; k++)
                    rectwidths[k]++;
                maxopen = h;
            }
            else
                maxopen = 0;
        }

        /* Hitting the end of row is the same as hitting empty space. */
        close_rectangles(rectwidths, 0, maxopen, &wmax, &hmax);
        remember_largest_rectangle(iboxes, gno, j, i, wmax, hmax);
    }

    g_free(rectwidths);
    g_free(colsizes);

    return iboxes;
}

/**
 * gwy_field_get_grain_sizes:
 * @mask_field: Data field containing positive values in grains, nonpositive in free space.  However its contents is
 *              ignored as all grain information is taken from @grains (its dimensions determine the dimensions of
 *              @grains).
 * @ngrains: The number of grains as returned by gwy_field_number_grains().
 * @grains: Grain numbers filled with gwy_field_number_grains().
 * @sizes: Array of size at least @ngrains+1 to fill with grain sizes (as usual zero does not correspond to any grain,
 *         grains start from 1). It can be %NULL to allocate a new array.
 *
 * Find sizes of all grains in a mask data field.
 *
 * Size is the number of pixels in the grain.
 *
 * The zeroth element of @sizes is filled with the number of pixels not covered by the mask.
 *
 * Returns: Either @sizes (if it was not %NULL), or a newly allocated array of size @ngrains+1.
 **/
gint*
gwy_field_get_grain_sizes(GwyField *mask_field,
                          gint ngrains,
                          const gint *grains,
                          gint *sizes)
{
    gint xres, yres, k;

    g_return_val_if_fail(GWY_IS_FIELD(mask_field), NULL);
    g_return_val_if_fail(grains, NULL);

    xres = mask_field->xres;
    yres = mask_field->yres;
    if (!sizes)
        sizes = g_new(gint, ngrains + 1);

    gwy_clear(sizes, ngrains + 1);
    for (k = 0; k < xres*yres; k++)
        sizes[grains[k]]++;

    return sizes;
}

/**
 * gwy_field_fill_grain:
 * @field: A data field with zeroes in empty space and nonzeroes in grains.
 * @col: Column inside a grain.
 * @row: Row inside a grain.
 * @nindices: Where the number of points in the grain at (@col, @row) should be stored.
 *
 * Finds all the points belonging to the grain at (@col, @row).
 *
 * Returns: A newly allocated array of indices of grain points in @dfield's data, the size of the list is returned in
 *          @nindices.
 **/
static gint*
gwy_field_fill_grain(GwyField *field,
                     gint col, gint row, gint *nindices)
{
    IntList *listv, *listh;
    gint *data, *visited;
    gint *indices;
    gint xres, yres, n, count;
    gint i, j;
    gint initial;

    xres = field->xres;
    yres = field->yres;
    initial = row*xres + col;
    gdouble *d = field->priv->data;
    g_return_val_if_fail(d[initial], NULL);

    /* check for a single point */
    if ((!col || d[initial - 1] <= 0)
        && (!row || d[initial - xres] <= 0)
        && (col + 1 == xres || d[initial + 1] <= 0)
        && (row + 1 == yres || d[initial + xres] <= 0)) {
        indices = g_new(gint, 1);

        indices[0] = initial;
        *nindices = 1;

        return indices;
    }

    n = xres*yres;
    visited = g_new0(gint, n);
    data = g_new(gint, n);
    listv = int_list_new(64);
    listh = int_list_new(64);

    for (i = 0; i < n; i++)
        data[i] = d[i] > 0;

    count = gwy_field_fill_one_grain(xres, yres, data, col, row, visited, 1, listv, listh);

    int_list_free(listh);
    int_list_free(listv);
    g_free(data);

    indices = g_new(gint, count);

    j = 0;
    for (i = 0; i < n; i++) {
        if (visited[i])
            indices[j++] = i;
    }
    g_free(visited);

    *nindices = count;
    return indices;
}

/**
 * gwy_field_fill_one_grain:
 * @xres: The number of columns in @data.
 * @yres: The number of rows in @data.
 * @data: Arbitrary integer data.  Grain is formed by values equal to the value at (@col, @row).
 * @col: Column inside a grain.
 * @row: Row inside a grain.
 * @visited: An array @col x @row that contain zeroes in empty space and yet unvisited grains.  Current grain will be
 *           filled with @grain_no.
 * @grain_no: Value to fill current grain with.
 * @listv: A working buffer of size at least @col x @row/2 + 2, its content is owerwritten.
 * @listh: A working buffer of size at least @col x @row/2 + 2, its content is owerwritten.
 *
 * Internal function to fill/number a one grain.
 *
 * The @visited, @listv, and @listh buffers are recyclable between calls so they don't have to be allocated and freed
 * for each grain, speeding up sequential grain processing.  Generally, this function itself does not allocate or free
 * any memory.
 *
 * Returns: The number of pixels in the grain.
 **/
static gint
gwy_field_fill_one_grain(gint xres,
                         gint yres,
                         const gint *data,
                         gint col, gint row,
                         gint *visited,
                         gint grain_no,
                         IntList *listv,
                         IntList *listh)
{
    gint n, count;
    gint i, p, j;
    gint initial;
    gint look_for;

    g_return_val_if_fail(grain_no, 0);
    initial = row*xres + col;
    look_for = data[initial];

    /* check for a single point */
    visited[initial] = grain_no;
    count = 1;
    if ((!col || data[initial - 1] != look_for)
        && (!row || data[initial - xres] != look_for)
        && (col + 1 == xres || data[initial + 1] != look_for)
        && (row + 1 == yres || data[initial + xres] != look_for)) {

        return count;
    }

    n = xres*yres;
    listv->len = listh->len = 0;
    int_list_add(listv, initial);
    int_list_add(listv, initial);
    int_list_add(listh, initial);
    int_list_add(listh, initial);

    while (listv->len) {
        /* go through vertical lines and expand them horizontally */
        for (i = 0; i < listv->len; i += 2) {
            for (p = listv->data[i]; p <= listv->data[i + 1]; p += xres) {
                gint start, stop;

                /* scan left */
                start = p - 1;
                stop = (p/xres)*xres;
                for (j = start; j >= stop; j--) {
                    if (visited[j] || data[j] != look_for)
                        break;
                    visited[j] = grain_no;
                    count++;
                }
                if (j < start) {
                    int_list_add(listh, j + 1);
                    int_list_add(listh, start);
                }

                /* scan right */
                start = p + 1;
                stop = (p/xres + 1)*xres;
                for (j = start; j < stop; j++) {
                    if (visited[j] || data[j] != look_for)
                        break;
                    visited[j] = grain_no;
                    count++;
                }
                if (j > start) {
                    int_list_add(listh, start);
                    int_list_add(listh, j - 1);
                }
            }
        }
        listv->len = 0;

        /* go through horizontal lines and expand them vertically */
        for (i = 0; i < listh->len; i += 2) {
            for (p = listh->data[i]; p <= listh->data[i + 1]; p++) {
                gint start, stop;

                /* scan up */
                start = p - xres;
                stop = p % xres;
                for (j = start; j >= stop; j -= xres) {
                    if (visited[j] || data[j] != look_for)
                        break;
                    visited[j] = grain_no;
                    count++;
                }
                if (j < start) {
                    int_list_add(listv, j + xres);
                    int_list_add(listv, start);
                }

                /* scan down */
                start = p + xres;
                stop = p % xres + n;
                for (j = start; j < stop; j += xres) {
                    if (visited[j] || data[j] != look_for)
                        break;
                    visited[j] = grain_no;
                    count++;
                }
                if (j > start) {
                    int_list_add(listv, start);
                    int_list_add(listv, j - xres);
                }
            }
        }
        listh->len = 0;
    }

    return count;
}

/**
 * gwy_field_fill_voids:
 * @field: A data field with zeroes in empty space and nonzeroes in grains.
 * @nonsimple: Pass %TRUE to fill also voids that are not simple-connected (e.g. ring-like).  This can result in grain
 *             merging if a small grain is contained within a void.  Pass %FALSE to fill only simple-connected grains.
 *
 * Fills voids in grains in a data field representing a mask.
 *
 * Voids in grains are zero pixels in @field from which no path exists through other zero pixels to the field
 * boundary.  The paths are considered in 8-connectivity because grains themselves are considered in 4-connectivity.
 *
 * Returns: %TRUE if any voids were filled at all, %FALSE if no change was made.
 **/
gboolean
gwy_field_fill_voids(GwyField *field,
                     gboolean nonsimple)
{
    GwyField *voids;
    gdouble *data;
    gint xres, yres;
    gint i, j, k, gno, gno2;
    gint *grains = NULL, *vgrains = NULL, *grain_neighbours = NULL;
    gboolean *unbound_vgrains;
    IntList **vgrain_neighbours;
    guint nvgrains;
    gboolean changed, changed_ever = FALSE, onlysimple = !nonsimple;

    g_return_val_if_fail(field, FALSE);

    xres = gwy_field_get_xres(field);
    yres = gwy_field_get_yres(field);
    data = gwy_field_get_data(field);

    voids = gwy_field_copy(field);
    gwy_field_grains_invert(voids);
    vgrains = g_new0(gint, xres*yres);
    nvgrains = gwy_field_number_grains(voids, vgrains);
    g_object_unref(voids);

    unbound_vgrains = g_new0(gboolean, nvgrains+1);
    for (i = 0; i < xres; i++) {
        unbound_vgrains[vgrains[i]] = TRUE;
        unbound_vgrains[vgrains[xres*(yres - 1) + i]] = TRUE;
    }
    for (j = 0; j < yres; j++) {
        unbound_vgrains[vgrains[j*xres]] = TRUE;
        unbound_vgrains[vgrains[j*xres + xres-1]] = TRUE;
    }

    if (onlysimple) {
        grains = g_new0(gint, xres*yres);
        grain_neighbours = g_new0(gint, nvgrains+1);
        gwy_field_number_grains(field, grains);
    }

    vgrain_neighbours = g_new0(IntList*, nvgrains+1);
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            k = i*xres + j;
            if (!(gno = vgrains[k]))
                continue;

            /* We must take into account grain separators (vgrains) have 8-connectivity while all grain functions work
             * with 4-connectivity.  So construct a map of diagonally touching grains and spread the unboundness
             * through it. */
            if (i && j && (gno2 = vgrains[k-xres-1]) && gno2 != gno)
                int_list_add_unique(vgrain_neighbours + gno, gno2);
            if (i && j < xres-1 && (gno2 = vgrains[k-xres+1]) && gno2 != gno)
                int_list_add_unique(vgrain_neighbours + gno, gno2);
            if (i < yres-1 && j && (gno2 = vgrains[k+xres-1]) && gno2 != gno)
                int_list_add_unique(vgrain_neighbours + gno, gno2);
            if (i < yres-1 && j < xres-1 && (gno2 = vgrains[k+xres+1]) && gno2 != gno)
                int_list_add_unique(vgrain_neighbours + gno, gno2);

            /* To detect non-simple-connected grains, we remember the number of the first normal grain the vgrain
             * touches.  If we find later that the void grain also touches a normal grain with a different number, it
             * means it is not simple-connected and we indicate by setting the neighbour number to G_MAXINT. */
            if (!onlysimple || grain_neighbours[gno] == G_MAXINT)
                continue;

            if (i && (gno2 = grains[k-xres])) {
                if (!grain_neighbours[gno])
                    grain_neighbours[gno] = gno2;
                else if (grain_neighbours[gno] != gno2)
                    grain_neighbours[gno] = G_MAXINT;
            }
            if (j && (gno2 = grains[k-1])) {
                if (!grain_neighbours[gno])
                    grain_neighbours[gno] = gno2;
                else if (grain_neighbours[gno] != gno2)
                    grain_neighbours[gno] = G_MAXINT;
            }
            if (j < xres-1 && (gno2 = grains[k+1])) {
                if (!grain_neighbours[gno])
                    grain_neighbours[gno] = gno2;
                else if (grain_neighbours[gno] != gno2)
                    grain_neighbours[gno] = G_MAXINT;
            }
            if (i < yres-1 && (gno2 = grains[k+xres])) {
                if (!grain_neighbours[gno])
                    grain_neighbours[gno] = gno2;
                else if (grain_neighbours[gno] != gno2)
                    grain_neighbours[gno] = G_MAXINT;
            }
        }
    }

    do {
        changed = FALSE;
        for (gno = 1; gno <= nvgrains; gno++) {
            IntList *list = vgrain_neighbours[gno];

            if (!list || !unbound_vgrains[gno])
                continue;

            changed = TRUE;
            for (k = 0; k < list->len; k++)
                unbound_vgrains[list->data[k]] = TRUE;

            /* We have propagated everything we can from @list, so avoid doing that again and again. */
            int_list_free(list);
            vgrain_neighbours[gno] = NULL;
        }

        changed_ever |= changed;
    } while (changed);

    k = 0;
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            if (!data[k] && !unbound_vgrains[vgrains[k]] && (!onlysimple || grain_neighbours[vgrains[k]] != G_MAXINT))
                data[k] = 1.0;
            k++;
        }
    }

    for (gno = 1; gno <= nvgrains; gno++) {
        IntList *list = vgrain_neighbours[gno];
        if (list)
            int_list_free(list);
    }
    g_free(vgrain_neighbours);
    g_free(grain_neighbours);
    g_free(unbound_vgrains);
    g_free(vgrains);
    g_free(grains);

    if (changed_ever)
        gwy_field_invalidate(field);

    return changed_ever;
}

static inline void
add_boundary_grain(IntList *bpixels, int *bindex, gint gno, gint k)
{
    int_list_add(bpixels, gno);
    int_list_add(bpixels, k);
    bindex[gno]++;
}

/**
 * gwy_field_grains_find_boundaries:
 * @field: Data field containing positive values in grains, nonpositive in free space.  However its contents is
 *              ignored as all grain information is taken from @grains (its dimensions determine the dimensions of
 *              @grains).
 * @grains: Grain numbers filled with gwy_field_number_grains().
 * @ngrains: The number of grains as returned by gwy_field_number_grains().
 * @bindex: Array of size at least @ngrains+2.  It will be filled block starts in the returned array.
 * @from_border: %TRUE to consider image edges to be grain boundaries.
 *
 * Find boundary pixels of all grains in a data field.
 *
 * The returned array contains pixel indices of grain boundaries, concatenated all one after another.  The block for
 * grain with number @i starts at position @bindex[@i] and ends one before position @bindex[@i+1] where the next block
 * starts.  The indices in each block are stored in ascending order.
 *
 * Boundary pixels are considered in the usual 4-connected metric.
 *
 * The boundary of the no-grain area is not computed. Therefore, the first two elements of @bindex will be always
 * zeros.
 *
 * Returns: A newly allocated array of size given by @bindex[@ngrains+1].
 **/
gint*
gwy_field_grains_find_boundaries(GwyField *field,
                                 const gint *grains,
                                 gint ngrains,
                                 gint *bindex,
                                 gboolean from_border)
{
    gint i, j, k, g;
    const gint *grow, *gprev, *gnext;
    IntList *bpixels;
    gint *retval;

    g_return_val_if_fail(field, NULL);
    g_return_val_if_fail(grains, NULL);
    g_return_val_if_fail(bindex, NULL);

    gint xres = field->xres, yres = field->yres;
    gwy_clear(bindex, ngrains+2);
    bpixels = int_list_new(ngrains+1);

    /* First row. */
    i = 0;
    grow = grains + i*xres;
    gnext = grow + xres;
    j = 0;
    if ((g = grow[j]) && (from_border | (g ^ grow[j+1]) | (g ^ gnext[j])))
        add_boundary_grain(bpixels, bindex, g, i*xres + j);
    for (j = 1; j < xres-1; j++) {
        if ((g = grow[j]) && (from_border | (g ^ grow[j-1]) | (g ^ grow[j+1]) | (g ^ gnext[j])))
            add_boundary_grain(bpixels, bindex, g, i*xres + j);
    }
    j = xres-1;
    if ((g = grow[j]) && (from_border | (g ^ grow[j-1]) | (g ^ gnext[j])))
        add_boundary_grain(bpixels, bindex, g, i*xres + j);

    /* Middle rows - we do not have to care abound borders, except for the first and last pixels. */
    for (i = 1; i < yres-1; i++) {
        grow = grains + i*xres;
        gprev = grow - xres;
        gnext = grow + xres;
        j = 0;
        if ((g = grow[j]) && (from_border | (g ^ grow[j+1]) | (g ^ gprev[j]) | (g ^ gnext[j])))
            add_boundary_grain(bpixels, bindex, g, i*xres + j);
        for (j = 1; j < xres-1; j++) {
            if ((g = grow[j]) && ((g ^ grow[j-1]) | (g ^ grow[j+1]) | (g ^ gprev[j]) | (g ^ gnext[j])))
                add_boundary_grain(bpixels, bindex, g, i*xres + j);
        }
        j = xres-1;
        if ((g = grow[j]) && (from_border | (g ^ grow[j-1]) | (g ^ gprev[j]) | (g ^ gnext[j])))
            add_boundary_grain(bpixels, bindex, g, i*xres + j);
    }

    /* Last row. */
    i = yres-1;
    grow = grains + i*xres;
    gprev = grow - xres;
    j = 0;
    if ((g = grow[j]) && (from_border | (g ^ grow[j+1]) | (g ^ gprev[j])))
        add_boundary_grain(bpixels, bindex, g, i*xres + j);
    for (j = 1; j < xres-1; j++) {
        if ((g = grow[j]) && (from_border | (g ^ grow[j-1]) | (g ^ grow[j+1]) | (g ^ gprev[j])))
            add_boundary_grain(bpixels, bindex, g, i*xres + j);
    }
    j = xres-1;
    if ((g = grow[j]) && (from_border | (g ^ grow[j-1]) | (g ^ gprev[j])))
        add_boundary_grain(bpixels, bindex, g, i*xres + j);

    /* Sum and rewind the index. */
    gwy_accumulate_counts(bindex, ngrains+1, TRUE);

    /* Fill the result using remembered bpixels. */
    retval = g_new(gint, bindex[ngrains+1]);
    for (i = 0; i < bpixels->len; i += 2) {
        g = bpixels->data[i];
        k = bpixels->data[i+1];
        retval[bindex[g]++] = k;
    }
    int_list_free(bpixels);
    for (i = ngrains+1; i > 0; i--)
        bindex[i] = bindex[i-1];
    bindex[0] = 0;

    return retval;
}

/**
 * SECTION:grains
 * @title: grains
 * @short_description: Grain detection and processing
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
