#!/bin/sh

# This tool installs the imip-agent software and message resources. It is
# configured by the contents of the config.sh script.
#
# Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program.  If not, see <http://www.gnu.org/licenses/>.

DIRNAME=`dirname "$0"`

if [ -e "$DIRNAME/config.sh" ]; then
    . "$DIRNAME/config.sh"
else
    . /etc/imip-agent/config.sh
fi

if [ "$1" = "--no-locale-dir" ]; then
    NO_LOCALE_DIR=$1
else
    NO_LOCALE_DIR=
fi

# Agents and modules.

AGENTS="imip_person.py imip_person_outgoing.py imip_resource.py"
MODULES="markup.py vCalendar.py vContent.py vRecurrence.py"

if [ ! -e "$INSTALL_DIR" ]; then
    mkdir -p "$INSTALL_DIR"
fi

cp $AGENTS "$INSTALL_DIR"
cp $MODULES "$INSTALL_DIR"

# Package modules.

for DIR in "$INSTALL_DIR/imiptools" \
           "$INSTALL_DIR/imiptools/stores" \
           "$INSTALL_DIR/imiptools/stores/database" \
           "$INSTALL_DIR/imiptools/handlers" \
           "$INSTALL_DIR/imiptools/handlers/scheduling" ; do
    if [ ! -e "$DIR" ]; then
        mkdir "$DIR"
    fi
done

# Remove any symbolic link to the config module. This linking is no longer
# supported, anyway.

if [ -h "$INSTALL_DIR/imiptools/config.py" ]; then
    rm "$INSTALL_DIR/imiptools/config.py"
fi

# Copy modules into the installation directory.

cp imiptools/*.py "$INSTALL_DIR/imiptools/"
cp imiptools/stores/*.py "$INSTALL_DIR/imiptools/stores/"
cp imiptools/stores/database/*.py "$INSTALL_DIR/imiptools/stores/database/"
cp imiptools/handlers/*.py "$INSTALL_DIR/imiptools/handlers/"
cp imiptools/handlers/scheduling/*.py "$INSTALL_DIR/imiptools/handlers/scheduling/"

# Remove migrated modules.

if [ -e "$INSTALL_DIR/imiptools/handlers/scheduling.py" ]; then
    rm "$INSTALL_DIR/imiptools/handlers/scheduling.py"*
fi

if [ -e "$INSTALL_DIR/imip_store.py" ]; then
    rm "$INSTALL_DIR/imip_store.py"*
fi

# Install the configuration in a more appropriate location.
# Create new versions of configuration files instead of overwriting.

if [ ! -e "$CONFIG_DIR" ]; then
    mkdir -p "$CONFIG_DIR"
fi

# Handle any old config module in the configuration location.

if [ -e "$CONFIG_DIR/config.py" ]; then

    # Rename the old config module.

    mv "$CONFIG_DIR/config.py" "$CONFIG_DIR/config.txt"
fi

# Handle any existing configuration file in the configuration location.

if [ -e "$CONFIG_DIR/config.txt" ]; then

    # Install the new configuration file alongside the existing file if
    # different.

    if ! cmp "imiptools/config.txt" "$CONFIG_DIR/config.txt" > /dev/null 2>&1 ; then
        cp "imiptools/config.txt" "$CONFIG_DIR/config.txt.new"
    fi

# Otherwise, just copy the configuration file.

else
    cp "imiptools/config.txt" "$CONFIG_DIR/config.txt"
fi

# Update the tools configuration file.

if [ -e "$CONFIG_DIR/config.sh" ]; then
    if ! cmp "tools/config.sh" "$CONFIG_DIR/config.sh" > /dev/null 2>&1 ; then
        cp "tools/config.sh" "$CONFIG_DIR/config.sh.new"
    fi
else
    cp "tools/config.sh" "$CONFIG_DIR/config.sh"
fi

# Copy related configuration files.

if [ ! -e "$CONFIG_DIR/postgresql" ]; then
    mkdir -p "$CONFIG_DIR/postgresql"
fi

if [ -e "$CONFIG_DIR/postgresql/schema.sql" ]; then
    if ! cmp "conf/postgresql/schema.sql" "$CONFIG_DIR/postgresql/schema.sql" > /dev/null 2>&1 ; then
        cp "conf/postgresql/schema.sql" "$CONFIG_DIR/postgresql/schema.sql.new"
    fi
else
    cp "conf/postgresql/schema.sql" "$CONFIG_DIR/postgresql/schema.sql"
fi

# Tools

TOOLS="copy_store.py fix.sh init.sh init_user.sh make_freebusy.py set_delegates.py "\
"set_quota_groups.py set_quota_limits.py update_quotas.py"

OLD_TOOLS="update_scheduling_modules.py update_storage_modules.py"

if [ ! -e "$INSTALL_DIR/tools" ]; then
    mkdir -p "$INSTALL_DIR/tools"
fi

for TOOL in $TOOLS; do
    cp "tools/$TOOL" "$INSTALL_DIR/tools/"
done

for TOOL in $OLD_TOOLS; do
    if [ -e "$INSTALL_DIR/tools/$TOOL" ]; then
        rm "$INSTALL_DIR/tools/$TOOL"
    fi
done

# Web manager interface.

if [ ! -e "$WEB_INSTALL_DIR" ]; then
    mkdir -p "$WEB_INSTALL_DIR"
fi

cp imip_manager.py "$WEB_INSTALL_DIR"
cp htdocs/styles.css "$WEB_INSTALL_DIR"

if [ ! -e "$WEB_INSTALL_DIR/imipweb" ]; then
    mkdir "$WEB_INSTALL_DIR/imipweb"
fi

cp imipweb/*.py "$WEB_INSTALL_DIR/imipweb/"

# Locale directory.

if [ ! "$NO_LOCALE_DIR" ]; then

    # Make the locale directory if it does not exist.

    if [ ! -e "locale" ]; then
        "tools/i18n_format.sh"
    fi

    # Only copy the translations if they do now exist.

    if [ -e "locale" ]; then
        for DIR in "locale/"*"/LC_MESSAGES" ; do
            mkdir -p "$INSTALL_DIR/$DIR"
            cp "$DIR/"*.mo "$INSTALL_DIR/$DIR/"
        done
    fi
fi
