/*
 * Copyright 2025 Valve Corporation
 * SPDX-License-Identifier: MIT
 */


#include <math.h>
#include <gtest/gtest.h>

#include "util/float8.h"
#include "u_math.h"

/* Navi48 output. */
static const uint32_t e4m3fn_lut[256] = {
   0x0,        /* 0x0 */
   0x3b000000, /* 0x1 */
   0x3b800000, /* 0x2 */
   0x3bc00000, /* 0x3 */
   0x3c000000, /* 0x4 */
   0x3c200000, /* 0x5 */
   0x3c400000, /* 0x6 */
   0x3c600000, /* 0x7 */
   0x3c800000, /* 0x8 */
   0x3c900000, /* 0x9 */
   0x3ca00000, /* 0xa */
   0x3cb00000, /* 0xb */
   0x3cc00000, /* 0xc */
   0x3cd00000, /* 0xd */
   0x3ce00000, /* 0xe */
   0x3cf00000, /* 0xf */
   0x3d000000, /* 0x10 */
   0x3d100000, /* 0x11 */
   0x3d200000, /* 0x12 */
   0x3d300000, /* 0x13 */
   0x3d400000, /* 0x14 */
   0x3d500000, /* 0x15 */
   0x3d600000, /* 0x16 */
   0x3d700000, /* 0x17 */
   0x3d800000, /* 0x18 */
   0x3d900000, /* 0x19 */
   0x3da00000, /* 0x1a */
   0x3db00000, /* 0x1b */
   0x3dc00000, /* 0x1c */
   0x3dd00000, /* 0x1d */
   0x3de00000, /* 0x1e */
   0x3df00000, /* 0x1f */
   0x3e000000, /* 0x20 */
   0x3e100000, /* 0x21 */
   0x3e200000, /* 0x22 */
   0x3e300000, /* 0x23 */
   0x3e400000, /* 0x24 */
   0x3e500000, /* 0x25 */
   0x3e600000, /* 0x26 */
   0x3e700000, /* 0x27 */
   0x3e800000, /* 0x28 */
   0x3e900000, /* 0x29 */
   0x3ea00000, /* 0x2a */
   0x3eb00000, /* 0x2b */
   0x3ec00000, /* 0x2c */
   0x3ed00000, /* 0x2d */
   0x3ee00000, /* 0x2e */
   0x3ef00000, /* 0x2f */
   0x3f000000, /* 0x30 */
   0x3f100000, /* 0x31 */
   0x3f200000, /* 0x32 */
   0x3f300000, /* 0x33 */
   0x3f400000, /* 0x34 */
   0x3f500000, /* 0x35 */
   0x3f600000, /* 0x36 */
   0x3f700000, /* 0x37 */
   0x3f800000, /* 0x38 */
   0x3f900000, /* 0x39 */
   0x3fa00000, /* 0x3a */
   0x3fb00000, /* 0x3b */
   0x3fc00000, /* 0x3c */
   0x3fd00000, /* 0x3d */
   0x3fe00000, /* 0x3e */
   0x3ff00000, /* 0x3f */
   0x40000000, /* 0x40 */
   0x40100000, /* 0x41 */
   0x40200000, /* 0x42 */
   0x40300000, /* 0x43 */
   0x40400000, /* 0x44 */
   0x40500000, /* 0x45 */
   0x40600000, /* 0x46 */
   0x40700000, /* 0x47 */
   0x40800000, /* 0x48 */
   0x40900000, /* 0x49 */
   0x40a00000, /* 0x4a */
   0x40b00000, /* 0x4b */
   0x40c00000, /* 0x4c */
   0x40d00000, /* 0x4d */
   0x40e00000, /* 0x4e */
   0x40f00000, /* 0x4f */
   0x41000000, /* 0x50 */
   0x41100000, /* 0x51 */
   0x41200000, /* 0x52 */
   0x41300000, /* 0x53 */
   0x41400000, /* 0x54 */
   0x41500000, /* 0x55 */
   0x41600000, /* 0x56 */
   0x41700000, /* 0x57 */
   0x41800000, /* 0x58 */
   0x41900000, /* 0x59 */
   0x41a00000, /* 0x5a */
   0x41b00000, /* 0x5b */
   0x41c00000, /* 0x5c */
   0x41d00000, /* 0x5d */
   0x41e00000, /* 0x5e */
   0x41f00000, /* 0x5f */
   0x42000000, /* 0x60 */
   0x42100000, /* 0x61 */
   0x42200000, /* 0x62 */
   0x42300000, /* 0x63 */
   0x42400000, /* 0x64 */
   0x42500000, /* 0x65 */
   0x42600000, /* 0x66 */
   0x42700000, /* 0x67 */
   0x42800000, /* 0x68 */
   0x42900000, /* 0x69 */
   0x42a00000, /* 0x6a */
   0x42b00000, /* 0x6b */
   0x42c00000, /* 0x6c */
   0x42d00000, /* 0x6d */
   0x42e00000, /* 0x6e */
   0x42f00000, /* 0x6f */
   0x43000000, /* 0x70 */
   0x43100000, /* 0x71 */
   0x43200000, /* 0x72 */
   0x43300000, /* 0x73 */
   0x43400000, /* 0x74 */
   0x43500000, /* 0x75 */
   0x43600000, /* 0x76 */
   0x43700000, /* 0x77 */
   0x43800000, /* 0x78 */
   0x43900000, /* 0x79 */
   0x43a00000, /* 0x7a */
   0x43b00000, /* 0x7b */
   0x43c00000, /* 0x7c */
   0x43d00000, /* 0x7d */
   0x43e00000, /* 0x7e */
   0xffc00000, /* 0x7f */
   0x80000000, /* 0x80 */
   0xbb000000, /* 0x81 */
   0xbb800000, /* 0x82 */
   0xbbc00000, /* 0x83 */
   0xbc000000, /* 0x84 */
   0xbc200000, /* 0x85 */
   0xbc400000, /* 0x86 */
   0xbc600000, /* 0x87 */
   0xbc800000, /* 0x88 */
   0xbc900000, /* 0x89 */
   0xbca00000, /* 0x8a */
   0xbcb00000, /* 0x8b */
   0xbcc00000, /* 0x8c */
   0xbcd00000, /* 0x8d */
   0xbce00000, /* 0x8e */
   0xbcf00000, /* 0x8f */
   0xbd000000, /* 0x90 */
   0xbd100000, /* 0x91 */
   0xbd200000, /* 0x92 */
   0xbd300000, /* 0x93 */
   0xbd400000, /* 0x94 */
   0xbd500000, /* 0x95 */
   0xbd600000, /* 0x96 */
   0xbd700000, /* 0x97 */
   0xbd800000, /* 0x98 */
   0xbd900000, /* 0x99 */
   0xbda00000, /* 0x9a */
   0xbdb00000, /* 0x9b */
   0xbdc00000, /* 0x9c */
   0xbdd00000, /* 0x9d */
   0xbde00000, /* 0x9e */
   0xbdf00000, /* 0x9f */
   0xbe000000, /* 0xa0 */
   0xbe100000, /* 0xa1 */
   0xbe200000, /* 0xa2 */
   0xbe300000, /* 0xa3 */
   0xbe400000, /* 0xa4 */
   0xbe500000, /* 0xa5 */
   0xbe600000, /* 0xa6 */
   0xbe700000, /* 0xa7 */
   0xbe800000, /* 0xa8 */
   0xbe900000, /* 0xa9 */
   0xbea00000, /* 0xaa */
   0xbeb00000, /* 0xab */
   0xbec00000, /* 0xac */
   0xbed00000, /* 0xad */
   0xbee00000, /* 0xae */
   0xbef00000, /* 0xaf */
   0xbf000000, /* 0xb0 */
   0xbf100000, /* 0xb1 */
   0xbf200000, /* 0xb2 */
   0xbf300000, /* 0xb3 */
   0xbf400000, /* 0xb4 */
   0xbf500000, /* 0xb5 */
   0xbf600000, /* 0xb6 */
   0xbf700000, /* 0xb7 */
   0xbf800000, /* 0xb8 */
   0xbf900000, /* 0xb9 */
   0xbfa00000, /* 0xba */
   0xbfb00000, /* 0xbb */
   0xbfc00000, /* 0xbc */
   0xbfd00000, /* 0xbd */
   0xbfe00000, /* 0xbe */
   0xbff00000, /* 0xbf */
   0xc0000000, /* 0xc0 */
   0xc0100000, /* 0xc1 */
   0xc0200000, /* 0xc2 */
   0xc0300000, /* 0xc3 */
   0xc0400000, /* 0xc4 */
   0xc0500000, /* 0xc5 */
   0xc0600000, /* 0xc6 */
   0xc0700000, /* 0xc7 */
   0xc0800000, /* 0xc8 */
   0xc0900000, /* 0xc9 */
   0xc0a00000, /* 0xca */
   0xc0b00000, /* 0xcb */
   0xc0c00000, /* 0xcc */
   0xc0d00000, /* 0xcd */
   0xc0e00000, /* 0xce */
   0xc0f00000, /* 0xcf */
   0xc1000000, /* 0xd0 */
   0xc1100000, /* 0xd1 */
   0xc1200000, /* 0xd2 */
   0xc1300000, /* 0xd3 */
   0xc1400000, /* 0xd4 */
   0xc1500000, /* 0xd5 */
   0xc1600000, /* 0xd6 */
   0xc1700000, /* 0xd7 */
   0xc1800000, /* 0xd8 */
   0xc1900000, /* 0xd9 */
   0xc1a00000, /* 0xda */
   0xc1b00000, /* 0xdb */
   0xc1c00000, /* 0xdc */
   0xc1d00000, /* 0xdd */
   0xc1e00000, /* 0xde */
   0xc1f00000, /* 0xdf */
   0xc2000000, /* 0xe0 */
   0xc2100000, /* 0xe1 */
   0xc2200000, /* 0xe2 */
   0xc2300000, /* 0xe3 */
   0xc2400000, /* 0xe4 */
   0xc2500000, /* 0xe5 */
   0xc2600000, /* 0xe6 */
   0xc2700000, /* 0xe7 */
   0xc2800000, /* 0xe8 */
   0xc2900000, /* 0xe9 */
   0xc2a00000, /* 0xea */
   0xc2b00000, /* 0xeb */
   0xc2c00000, /* 0xec */
   0xc2d00000, /* 0xed */
   0xc2e00000, /* 0xee */
   0xc2f00000, /* 0xef */
   0xc3000000, /* 0xf0 */
   0xc3100000, /* 0xf1 */
   0xc3200000, /* 0xf2 */
   0xc3300000, /* 0xf3 */
   0xc3400000, /* 0xf4 */
   0xc3500000, /* 0xf5 */
   0xc3600000, /* 0xf6 */
   0xc3700000, /* 0xf7 */
   0xc3800000, /* 0xf8 */
   0xc3900000, /* 0xf9 */
   0xc3a00000, /* 0xfa */
   0xc3b00000, /* 0xfb */
   0xc3c00000, /* 0xfc */
   0xc3d00000, /* 0xfd */
   0xc3e00000, /* 0xfe */
   0xffc00000, /* 0xff */
};

static const uint32_t e5m2_lut[256] = {
   0x0,        /* 0x0 */
   0x37800000, /* 0x1 */
   0x38000000, /* 0x2 */
   0x38400000, /* 0x3 */
   0x38800000, /* 0x4 */
   0x38a00000, /* 0x5 */
   0x38c00000, /* 0x6 */
   0x38e00000, /* 0x7 */
   0x39000000, /* 0x8 */
   0x39200000, /* 0x9 */
   0x39400000, /* 0xa */
   0x39600000, /* 0xb */
   0x39800000, /* 0xc */
   0x39a00000, /* 0xd */
   0x39c00000, /* 0xe */
   0x39e00000, /* 0xf */
   0x3a000000, /* 0x10 */
   0x3a200000, /* 0x11 */
   0x3a400000, /* 0x12 */
   0x3a600000, /* 0x13 */
   0x3a800000, /* 0x14 */
   0x3aa00000, /* 0x15 */
   0x3ac00000, /* 0x16 */
   0x3ae00000, /* 0x17 */
   0x3b000000, /* 0x18 */
   0x3b200000, /* 0x19 */
   0x3b400000, /* 0x1a */
   0x3b600000, /* 0x1b */
   0x3b800000, /* 0x1c */
   0x3ba00000, /* 0x1d */
   0x3bc00000, /* 0x1e */
   0x3be00000, /* 0x1f */
   0x3c000000, /* 0x20 */
   0x3c200000, /* 0x21 */
   0x3c400000, /* 0x22 */
   0x3c600000, /* 0x23 */
   0x3c800000, /* 0x24 */
   0x3ca00000, /* 0x25 */
   0x3cc00000, /* 0x26 */
   0x3ce00000, /* 0x27 */
   0x3d000000, /* 0x28 */
   0x3d200000, /* 0x29 */
   0x3d400000, /* 0x2a */
   0x3d600000, /* 0x2b */
   0x3d800000, /* 0x2c */
   0x3da00000, /* 0x2d */
   0x3dc00000, /* 0x2e */
   0x3de00000, /* 0x2f */
   0x3e000000, /* 0x30 */
   0x3e200000, /* 0x31 */
   0x3e400000, /* 0x32 */
   0x3e600000, /* 0x33 */
   0x3e800000, /* 0x34 */
   0x3ea00000, /* 0x35 */
   0x3ec00000, /* 0x36 */
   0x3ee00000, /* 0x37 */
   0x3f000000, /* 0x38 */
   0x3f200000, /* 0x39 */
   0x3f400000, /* 0x3a */
   0x3f600000, /* 0x3b */
   0x3f800000, /* 0x3c */
   0x3fa00000, /* 0x3d */
   0x3fc00000, /* 0x3e */
   0x3fe00000, /* 0x3f */
   0x40000000, /* 0x40 */
   0x40200000, /* 0x41 */
   0x40400000, /* 0x42 */
   0x40600000, /* 0x43 */
   0x40800000, /* 0x44 */
   0x40a00000, /* 0x45 */
   0x40c00000, /* 0x46 */
   0x40e00000, /* 0x47 */
   0x41000000, /* 0x48 */
   0x41200000, /* 0x49 */
   0x41400000, /* 0x4a */
   0x41600000, /* 0x4b */
   0x41800000, /* 0x4c */
   0x41a00000, /* 0x4d */
   0x41c00000, /* 0x4e */
   0x41e00000, /* 0x4f */
   0x42000000, /* 0x50 */
   0x42200000, /* 0x51 */
   0x42400000, /* 0x52 */
   0x42600000, /* 0x53 */
   0x42800000, /* 0x54 */
   0x42a00000, /* 0x55 */
   0x42c00000, /* 0x56 */
   0x42e00000, /* 0x57 */
   0x43000000, /* 0x58 */
   0x43200000, /* 0x59 */
   0x43400000, /* 0x5a */
   0x43600000, /* 0x5b */
   0x43800000, /* 0x5c */
   0x43a00000, /* 0x5d */
   0x43c00000, /* 0x5e */
   0x43e00000, /* 0x5f */
   0x44000000, /* 0x60 */
   0x44200000, /* 0x61 */
   0x44400000, /* 0x62 */
   0x44600000, /* 0x63 */
   0x44800000, /* 0x64 */
   0x44a00000, /* 0x65 */
   0x44c00000, /* 0x66 */
   0x44e00000, /* 0x67 */
   0x45000000, /* 0x68 */
   0x45200000, /* 0x69 */
   0x45400000, /* 0x6a */
   0x45600000, /* 0x6b */
   0x45800000, /* 0x6c */
   0x45a00000, /* 0x6d */
   0x45c00000, /* 0x6e */
   0x45e00000, /* 0x6f */
   0x46000000, /* 0x70 */
   0x46200000, /* 0x71 */
   0x46400000, /* 0x72 */
   0x46600000, /* 0x73 */
   0x46800000, /* 0x74 */
   0x46a00000, /* 0x75 */
   0x46c00000, /* 0x76 */
   0x46e00000, /* 0x77 */
   0x47000000, /* 0x78 */
   0x47200000, /* 0x79 */
   0x47400000, /* 0x7a */
   0x47600000, /* 0x7b */
   0x7f800000, /* 0x7c */
   0xffc00000, /* 0x7d */
   0xffc00000, /* 0x7e */
   0xffc00000, /* 0x7f */
   0x80000000, /* 0x80 */
   0xb7800000, /* 0x81 */
   0xb8000000, /* 0x82 */
   0xb8400000, /* 0x83 */
   0xb8800000, /* 0x84 */
   0xb8a00000, /* 0x85 */
   0xb8c00000, /* 0x86 */
   0xb8e00000, /* 0x87 */
   0xb9000000, /* 0x88 */
   0xb9200000, /* 0x89 */
   0xb9400000, /* 0x8a */
   0xb9600000, /* 0x8b */
   0xb9800000, /* 0x8c */
   0xb9a00000, /* 0x8d */
   0xb9c00000, /* 0x8e */
   0xb9e00000, /* 0x8f */
   0xba000000, /* 0x90 */
   0xba200000, /* 0x91 */
   0xba400000, /* 0x92 */
   0xba600000, /* 0x93 */
   0xba800000, /* 0x94 */
   0xbaa00000, /* 0x95 */
   0xbac00000, /* 0x96 */
   0xbae00000, /* 0x97 */
   0xbb000000, /* 0x98 */
   0xbb200000, /* 0x99 */
   0xbb400000, /* 0x9a */
   0xbb600000, /* 0x9b */
   0xbb800000, /* 0x9c */
   0xbba00000, /* 0x9d */
   0xbbc00000, /* 0x9e */
   0xbbe00000, /* 0x9f */
   0xbc000000, /* 0xa0 */
   0xbc200000, /* 0xa1 */
   0xbc400000, /* 0xa2 */
   0xbc600000, /* 0xa3 */
   0xbc800000, /* 0xa4 */
   0xbca00000, /* 0xa5 */
   0xbcc00000, /* 0xa6 */
   0xbce00000, /* 0xa7 */
   0xbd000000, /* 0xa8 */
   0xbd200000, /* 0xa9 */
   0xbd400000, /* 0xaa */
   0xbd600000, /* 0xab */
   0xbd800000, /* 0xac */
   0xbda00000, /* 0xad */
   0xbdc00000, /* 0xae */
   0xbde00000, /* 0xaf */
   0xbe000000, /* 0xb0 */
   0xbe200000, /* 0xb1 */
   0xbe400000, /* 0xb2 */
   0xbe600000, /* 0xb3 */
   0xbe800000, /* 0xb4 */
   0xbea00000, /* 0xb5 */
   0xbec00000, /* 0xb6 */
   0xbee00000, /* 0xb7 */
   0xbf000000, /* 0xb8 */
   0xbf200000, /* 0xb9 */
   0xbf400000, /* 0xba */
   0xbf600000, /* 0xbb */
   0xbf800000, /* 0xbc */
   0xbfa00000, /* 0xbd */
   0xbfc00000, /* 0xbe */
   0xbfe00000, /* 0xbf */
   0xc0000000, /* 0xc0 */
   0xc0200000, /* 0xc1 */
   0xc0400000, /* 0xc2 */
   0xc0600000, /* 0xc3 */
   0xc0800000, /* 0xc4 */
   0xc0a00000, /* 0xc5 */
   0xc0c00000, /* 0xc6 */
   0xc0e00000, /* 0xc7 */
   0xc1000000, /* 0xc8 */
   0xc1200000, /* 0xc9 */
   0xc1400000, /* 0xca */
   0xc1600000, /* 0xcb */
   0xc1800000, /* 0xcc */
   0xc1a00000, /* 0xcd */
   0xc1c00000, /* 0xce */
   0xc1e00000, /* 0xcf */
   0xc2000000, /* 0xd0 */
   0xc2200000, /* 0xd1 */
   0xc2400000, /* 0xd2 */
   0xc2600000, /* 0xd3 */
   0xc2800000, /* 0xd4 */
   0xc2a00000, /* 0xd5 */
   0xc2c00000, /* 0xd6 */
   0xc2e00000, /* 0xd7 */
   0xc3000000, /* 0xd8 */
   0xc3200000, /* 0xd9 */
   0xc3400000, /* 0xda */
   0xc3600000, /* 0xdb */
   0xc3800000, /* 0xdc */
   0xc3a00000, /* 0xdd */
   0xc3c00000, /* 0xde */
   0xc3e00000, /* 0xdf */
   0xc4000000, /* 0xe0 */
   0xc4200000, /* 0xe1 */
   0xc4400000, /* 0xe2 */
   0xc4600000, /* 0xe3 */
   0xc4800000, /* 0xe4 */
   0xc4a00000, /* 0xe5 */
   0xc4c00000, /* 0xe6 */
   0xc4e00000, /* 0xe7 */
   0xc5000000, /* 0xe8 */
   0xc5200000, /* 0xe9 */
   0xc5400000, /* 0xea */
   0xc5600000, /* 0xeb */
   0xc5800000, /* 0xec */
   0xc5a00000, /* 0xed */
   0xc5c00000, /* 0xee */
   0xc5e00000, /* 0xef */
   0xc6000000, /* 0xf0 */
   0xc6200000, /* 0xf1 */
   0xc6400000, /* 0xf2 */
   0xc6600000, /* 0xf3 */
   0xc6800000, /* 0xf4 */
   0xc6a00000, /* 0xf5 */
   0xc6c00000, /* 0xf6 */
   0xc6e00000, /* 0xf7 */
   0xc7000000, /* 0xf8 */
   0xc7200000, /* 0xf9 */
   0xc7400000, /* 0xfa */
   0xc7600000, /* 0xfb */
   0xff800000, /* 0xfc */
   0xffc00000, /* 0xfd */
   0xffc00000, /* 0xfe */
   0xffc00000, /* 0xff */
};

TEST(float8_test, e4m3fn_to_float)
{
   for (unsigned i = 0; i < 256; i++) {
      EXPECT_EQ(fui(_mesa_e4m3fn_to_float(i)), e4m3fn_lut[i]);
   }
}

TEST(float8_test, e5m2_to_float)
{
   for (unsigned i = 0; i < 256; i++) {
      EXPECT_EQ(fui(_mesa_e5m2_to_float(i)), e5m2_lut[i]);
   }
}

TEST(float8_test, float_to_e4m3fn_exact)
{
   for (unsigned i = 0; i < 256; i++) {
      EXPECT_EQ(fui(_mesa_e4m3fn_to_float(_mesa_float_to_e4m3fn(uif(e4m3fn_lut[i])))), e4m3fn_lut[i]);
   }
}

TEST(float8_test, float_to_e5m2_exact)
{
   for (unsigned i = 0; i < 256; i++) {
      EXPECT_EQ(fui(_mesa_e5m2_to_float(_mesa_float_to_e5m2(uif(e5m2_lut[i])))), e5m2_lut[i]);
   }
}

TEST(float8_test, float_to_e4m3fn_rtne)
{
   /* underflow border */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3A800000)), 0);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3A800001)), 1);

   /* rounding up border, denorm */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3b3fffff)), 1);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3b400000)), 2);

   /* rounding down border, denorm */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3ba00000)), 2);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x3ba00001)), 3);

   /* rounding up border, normal */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x4017ffff)), 0x41);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x40180000)), 0x42);

   /* rounding down border, normal */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x40080000)), 0x40);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x40080001)), 0x41);

   /* overflow border */
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x43e80000)), 0x7e);
   EXPECT_EQ(_mesa_float_to_e4m3fn(uif(0x43e80001)), 0xff);
}

TEST(float8_test, float_to_e5m2_rtne)
{
   /* underflow border */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x37000000)), 0);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x37000001)), 1);

   /* rounding up border, denorm */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x37bfffff)), 1);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x37c00000)), 2);

   /* rounding down border, denorm */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x38200000)), 2);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x38200001)), 3);

   /* rounding up border, normal */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x402fffff)), 0x41);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x40300000)), 0x42);

   /* rounding down border, normal */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x40100000)), 0x40);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x40100001)), 0x41);

   /* overflow border */
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x476fffff)), 0x7b);
   EXPECT_EQ(_mesa_float_to_e5m2(uif(0x47700000)), 0x7c);
}
