/***********************************************/
/**
* @file gnssAntennaDefinitionCreate.cpp
*
* @brief Create GNSS antenna definition file.
*
* @author Torsten Mayer-Guerr
* @date 2019-08-29
*/
/***********************************************/

// Latex documentation
#define DOCSTRING docstring
static const char *docstring = R"(
Create a \file{GNSS antenna definition file}{gnssAntennaDefinition} (Antenna Center Variations, ACV) consisting of multiple antennas.
The antennas can be created from scratch or can be selected from existing files.
This program can also be used to modify existing files.

Furthermore it can be used to create accuracy definition files containing azimuth and elevation dependent accuracy values for antennas.
To create an accuracy pattern for phase observations with \verb|1 mm| accuracy at zenith and no azimuth dependency, define a
pattern with \config{type}=\verb|L|, \config{values}=\verb|0.001/cos(zenith/rho)|.

The antennas in \configFile{outputfileAntennaDefinition}{gnssAntennaDefinition}
are sorted by names and duplicates are removed (first one is kept).
)";

/***********************************************/

#include "programs/program.h"
#include "base/string.h"
#include "config/configRegister.h"
#include "files/fileGnssAntennaDefinition.h"
#include "files/filePlatform.h"

/***** CLASS ***********************************/

/** @brief Create GNSS antenna definition file.
* @ingroup programsGroup */
class GnssAntennaDefinitionCreate
{
public:
  void run(Config &config, Parallel::CommunicatorPtr comm);
};

GROOPS_REGISTER_PROGRAM(GnssAntennaDefinitionCreate, SINGLEPROCESS, "Create GNSS antenna definition file.", Gnss)

/***********************************************/

static Bool readConfig(Config &config, const std::string &name, GnssAntennaPattern &var, Config::Appearance mustSet, const std::string &defaultValue, const std::string &annotation)
{
  try
  {
    if(!readConfigSequence(config, name, mustSet, defaultValue, annotation))
      return FALSE;

    Angle maxZenith, dAzimuth;
    ExpressionVariablePtr valueExpr;

    readConfig(config, "type",         var.type,       Config::MUSTSET,  "",   "pattern matching of observation types");
    readConfig(config, "offsetX",      var.offset.x(), Config::DEFAULT,  "0",  "[m] antenna center offset");
    readConfig(config, "offsetY",      var.offset.y(), Config::DEFAULT,  "0",  "[m] antenna center offset");
    readConfig(config, "offsetZ",      var.offset.z(), Config::DEFAULT,  "0",  "[m] antenna center offset");
    readConfig(config, "deltaAzimuth", dAzimuth,       Config::MUSTSET,  "5",  "[degree] step size");
    readConfig(config, "deltaZenith",  var.dZenit,     Config::MUSTSET,  "5",  "[degree] step size");
    readConfig(config, "maxZenith",    maxZenith,      Config::DEFAULT,  "90", "[degree]");
    readConfig(config, "values",       valueExpr,      Config::OPTIONAL, "0",  "[m] expression (zenith, azimuth: variables)");
    endSequence(config);
    if(isCreateSchema(config))
      return TRUE;

    UInt rows   = static_cast<UInt>(std::round(2*PI/dAzimuth));
    UInt cols   = static_cast<UInt>(std::round(maxZenith/var.dZenit))+1;
    var.pattern = Matrix(rows,cols);

    if(valueExpr)
    {
      VariableList varList;
      varList.undefineVariable("zenith");
      varList.undefineVariable("azimuth");
      valueExpr->simplify(varList);

      for(UInt i=0; i<var.pattern.rows(); i++)
        for(UInt k=0; k<var.pattern.columns(); k++)
        {
          varList.setVariable("azimuth", i * 360./var.pattern.rows());
          varList.setVariable("zenith",  k * RAD2DEG * Double(var.dZenit));
          var.pattern(i, k) = valueExpr->evaluate(varList);
        }
    }

    return TRUE;
  }
  catch(std::exception &e)
  {
    GROOPS_RETHROW(e)
  }
}

/***********************************************/
/***** CLASS ***********************************/
/***********************************************/

// Latex documentation
static const char *docstringGnssAntennaDefintionList = R"(
\section{GnssAntennaDefintionList}\label{gnssAntennaDefintionListType}
Provides a list of GnssAntennaDefinitions as used in \program{GnssAntennaDefinitionCreate}.
)";

class GnssAntennaDefintionList;
typedef std::shared_ptr<GnssAntennaDefintionList> GnssAntennaDefintionListPtr;

class GnssAntennaDefintionList
{
protected:
  static Bool match(GnssType type1, GnssType type2)
  {
    return (type1 == type2) &&
           (!type1.hasWildcard(GnssType::PRN)       || type2.hasWildcard(GnssType::PRN))       &&
           (!type1.hasWildcard(GnssType::SYSTEM)    || type2.hasWildcard(GnssType::SYSTEM))    &&
           (!type1.hasWildcard(GnssType::FREQUENCY) || type2.hasWildcard(GnssType::FREQUENCY)) &&
           (!type1.hasWildcard(GnssType::TYPE)      || type2.hasWildcard(GnssType::TYPE))      &&
           (!type1.hasWildcard(GnssType::ATTRIBUTE) || type2.hasWildcard(GnssType::ATTRIBUTE)) &&
           (!type1.hasWildcard(GnssType::FREQ_NO)   || type2.hasWildcard(GnssType::FREQ_NO));
  }

public:
  std::vector<GnssAntennaDefinitionPtr> antennas;

  /// Creates an derived instance of this class.
  static GnssAntennaDefintionListPtr create(Config &config, const std::string &name);
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListNew = R"(
\subsection{New}
Creates a new antenna.
)";

class GnssAntennaDefintionListNew : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListNew(Config &config)
  {
    try
    {
      antennas.push_back(GnssAntennaDefinitionPtr(new GnssAntennaDefinition()));
      readConfig(config, "name",    antennas.back()->name,     Config::OPTIONAL, "", "");
      readConfig(config, "serial",  antennas.back()->serial,   Config::OPTIONAL, "", "");
      readConfig(config, "radome",  antennas.back()->radome,   Config::OPTIONAL, "", "");
      readConfig(config, "comment", antennas.back()->comment,  Config::OPTIONAL, "", "");
      readConfig(config, "pattern", antennas.back()->patterns, Config::MUSTSET,  "", "");
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListFromFile = R"(
\subsection{FromFile}
Select all or the first antenna from an \file{antenna definition file}{gnssAntennaDefinition}
which matches the wildcards.
)";

class GnssAntennaDefintionListFromFile : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListFromFile(Config &config)
  {
    try
    {
      FileName    fileNameAntenna;
      std::string name, serial, radome;
      Bool        onlyFirstMatch;

      readConfig(config, "inputfileAntennaDefinition", fileNameAntenna, Config::MUSTSET,  "",  "");
      readConfig(config, "name",                       name,            Config::OPTIONAL, "*", "");
      readConfig(config, "serial",                     serial,          Config::OPTIONAL, "*", "");
      readConfig(config, "radome",                     radome,          Config::OPTIONAL, "*", "");
      readConfig(config, "onlyFirstMatch",             onlyFirstMatch,  Config::DEFAULT,  "0", "otherwise all machting antennas included");
      if(isCreateSchema(config)) return;

      std::vector<GnssAntennaDefinitionPtr> antennasFile;
      readFileGnssAntennaDefinition(fileNameAntenna, antennasFile);

      std::regex pattern = String::wildcard2regex(GnssAntennaDefinition::str(name, serial, radome));
      for(auto &antenna : antennasFile)
        if(std::regex_match(antenna->str(), pattern))
        {
          antennas.push_back(antenna);
          if(onlyFirstMatch)
            break;
        }
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListFromStationInfo = R"(
\subsection{FromStationInfo}
Select all antennas from an \file{antenna definition file}{gnssAntennaDefinition}
which are used by a station within a defined time interval.
With \config{specializeAntenna} an individual antenna is created for each different serial number
using the general type specific values from file.
)";

class GnssAntennaDefintionListFromStationInfo : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListFromStationInfo(Config &config)
  {
    try
    {
      FileName fileNameStationInfo, fileNameAntenna;
      Time     timeStart, timeEnd = date2time(2500,1,1);
      Bool     specialize;

      readConfig(config, "inputfileStationInfo",       fileNameStationInfo, Config::MUSTSET,  "",  "");
      readConfig(config, "inputfileAntennaDefinition", fileNameAntenna,     Config::MUSTSET,  "",  "");
      readConfig(config, "timeStart",                  timeStart,           Config::OPTIONAL, "",  "only antennas used in this time interval");
      readConfig(config, "timeEnd",                    timeEnd,             Config::OPTIONAL, "",  "only antennas used in this time interval");
      readConfig(config, "specializeAntenna",          specialize,          Config::DEFAULT,  "0", "e.g. separate different serial numbers from stationInfo");
      if(isCreateSchema(config)) return;

      Platform stationInfo;
      std::vector<GnssAntennaDefinitionPtr> antennasFile;
      readFilePlatform(fileNameStationInfo, stationInfo);
      readFileGnssAntennaDefinition(fileNameAntenna, antennasFile);
      stationInfo.fillGnssAntennaDefinition(antennasFile);

      for(const auto &instrument : stationInfo.equipments)
      {
        auto antennaInfo = std::dynamic_pointer_cast<PlatformGnssAntenna>(instrument);
        if(antennaInfo && antennaInfo->timeEnd >= timeStart && antennaInfo->timeStart < timeEnd)
        {
          if(antennaInfo->antennaDef)
          {
            antennas.push_back(std::make_shared<GnssAntennaDefinition>(*antennaInfo->antennaDef));
            if(specialize)
            {
              if(antennas.back()->name.empty())   antennas.back()->name   = antennaInfo->name;
              if(antennas.back()->serial.empty()) antennas.back()->serial = antennaInfo->serial;
              if(antennas.back()->radome.empty()) antennas.back()->radome = antennaInfo->radome;
            }
          }
          else
            logWarning<<stationInfo.markerName<<"."<<stationInfo.markerNumber<<": no antenna definition found for "<<antennaInfo->str()<<Log::endl;
        }
      }
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListResample = R"(
\subsection{Resample}
The azimuth and elevation dependent antenna center variations (patterns) of all \config{antenna}s
are resampled to a new resolution.
)";

class GnssAntennaDefintionListResample : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListResample(Config &config)
  {
    try
    {
      std::vector<GnssAntennaDefintionListPtr> antennaLists;
      Angle dAzimuth (NAN_EXPR);
      Angle dZenith  (NAN_EXPR);
      Angle maxZenith(NAN_EXPR);

      readConfig(config, "antenna",      antennaLists, Config::MUSTSET,   "", "");
      readConfig(config, "deltaAzimuth", dAzimuth,     Config::OPTIONAL, "", "[degree] step size, empty: no change");
      readConfig(config, "deltaZenith",  dZenith,      Config::OPTIONAL, "", "[degree] step size, empty: no change");
      readConfig(config, "maxZenith",    maxZenith,    Config::OPTIONAL, "", "[degree], empty: no change");
      if(isCreateSchema(config)) return;

      for(auto &antennaList : antennaLists)
        for(auto &antenna : antennaList->antennas)
          for(auto &pattern : antenna->patterns)
          {
            Double dz   = (dZenith   > 0) ? dZenith   : pattern.dZenit;
            Double maxz = (maxZenith >=0) ? maxZenith : ((pattern.pattern.columns()-1) * Double(pattern.dZenit));
            UInt cols   = static_cast<UInt>(std::round(maxz/dz))+1;
            UInt rows   = (dAzimuth  > 0) ? static_cast<UInt>(std::round(2*PI/dAzimuth)) : pattern.pattern.rows();

            if((pattern.dZenit != dz) || (pattern.pattern.rows() != rows) || (pattern.pattern.columns() != cols))
            {
              Matrix acv(rows, cols);
              for(UInt i=0; i<rows; i++)
                for(UInt k=0; k<cols; k++)
                  acv(i, k) += pattern.antennaVariations(Angle(2*PI*i/rows), Angle(PI/2-k*dz), FALSE);
              pattern.dZenit  = dz;
              pattern.pattern = acv;
            }
          }

      for(auto &antennaList : antennaLists)
        antennas.insert(antennas.end(), antennaList->antennas.begin(), antennaList->antennas.end());
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListTransform = R"(
\subsection{Transform}
This class can be used to separate general antenna patterns for different \configClass{gnssType}{gnssType}s.
If the \config{antenna}s contain only one pattern for all GPS observations on the L1 frequency (\verb|*1*G**|),
the \config{patternTypes}=\verb|C1*G**| and \verb|L1*G**| create two patterns with the \verb|*1*G**| patterm as template.
The first matching pattern in the \config{antenna} is used as template.
Also new \config{additionalPattern} can be added (e.g. for \verb|*5*G**|).
With \config{addExistingPatterns} all already existing patterns that don't match completely to any of the above are added.
)";

class GnssAntennaDefintionListTransform : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListTransform(Config &config)
  {
    try
    {
      std::vector<GnssAntennaDefintionListPtr> antennaLists;
      std::vector<GnssType>                    types;
      std::vector<GnssAntennaPattern>          patternsAdd;
      Bool                                     addExistingPatterns;

      readConfig(config, "antenna",             antennaLists,        Config::MUSTSET,  "", "");
      readConfig(config, "patternTypes",        types,               Config::OPTIONAL, "",  "gnssType for each pattern (first match is used)");
      readConfig(config, "additionalPattern",   patternsAdd,         Config::OPTIONAL, "",  "additional new patterns");
      readConfig(config, "addExistingPatterns", addExistingPatterns, Config::DEFAULT,  "1", "add existing patterns that don't match completely any of the above");
      if(isCreateSchema(config)) return;

      for(auto &antennaList : antennaLists)
        for(auto &antenna : antennaList->antennas)
        {
          std::vector<GnssAntennaPattern> patternsOld = antenna->patterns;
          antenna->patterns.clear();
          for(GnssType type : types)
          {
            Bool found = FALSE;
            for(auto &patternOld : patternsOld)
              if(type == patternOld.type)
              {
                antenna->patterns.push_back(patternOld);
                antenna->patterns.back().type = type;
                found = TRUE;
                break;
              }
            if(!found)
              throw(Exception(antenna->str()+": no pattern found which fits to all possible types: "+type.str()));
          }

          for(auto &patternAdd : patternsAdd)
          {
            Bool found = FALSE;
            for(auto &patternNew : antenna->patterns)
              if(match(patternAdd.type, patternNew.type))
              {
                found = TRUE;
                break;
              }
            if(!found)
              antenna->patterns.push_back(patternAdd);
          }

          if(addExistingPatterns)
            for(auto &patternOld : patternsOld)
            {
              Bool found = FALSE;
              for(auto &patternNew : antenna->patterns)
                if(match(patternOld.type, patternNew.type))
                {
                  found = TRUE;
                  break;
                }
              if(!found)
                antenna->patterns.push_back(patternOld);
            }

          std::sort(antenna->patterns.begin(), antenna->patterns.end(), [](GnssAntennaPattern &a, GnssAntennaPattern &b){return a.type < b.type;});
        } // for(antennas)

      for(auto &antennaList : antennaLists)
        antennas.insert(antennas.end(), antennaList->antennas.begin(), antennaList->antennas.end());
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListRename = R"(
\subsection{Rename}
Replaces parts of the descrption of \config{antenna}s.
The star "\verb|*|" left this part untouched.
)";

class GnssAntennaDefintionListRename : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListRename(Config &config)
  {
    try
    {
      std::vector<GnssAntennaDefintionListPtr> antennaLists;
      std::string name, serial, radome, comment;

      readConfig(config, "antenna", antennaLists, Config::MUSTSET,  "",  "");
      readConfig(config, "name",    name,         Config::OPTIONAL, "*", "*: left this part untouched");
      readConfig(config, "serial",  serial,       Config::OPTIONAL, "*", "*: left this part untouched");
      readConfig(config, "radome",  radome,       Config::OPTIONAL, "*", "*: left this part untouched");
      readConfig(config, "comment", comment,      Config::OPTIONAL, "*", "*: left this part untouched");
      if(isCreateSchema(config)) return;

      for(auto &antennaList : antennaLists)
        for(auto &antenna : antennaList->antennas)
        {
          if(name    != "*") antenna->name    = name;
          if(serial  != "*") antenna->serial  = serial;
          if(radome  != "*") antenna->radome  = radome;
          if(comment != "*") antenna->comment = comment;
          antennas.push_back(antenna);
        } // for(antennas)
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListSetZero = R"(
\subsection{SetZero}
The antenna center variations (patterns) or offsets
of all \config{antenna}s are set to zero.
)";

class GnssAntennaDefintionListSetZero : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListSetZero(Config &config)
  {
    try
    {
      std::vector<GnssAntennaDefintionListPtr> antennaLists;
      std::vector<GnssType> types;
      Bool zeroPattern, zeroOffset;

      readConfig(config, "antenna",      antennaLists, Config::MUSTSET,  "",  "");
      readConfig(config, "patternTypes", types,        Config::OPTIONAL, "",  "only matching patterns, default: all");
      readConfig(config, "zeroOffset",   zeroOffset,   Config::DEFAULT,  "1", "");
      readConfig(config, "zeroPattern",  zeroPattern,  Config::DEFAULT,  "1", "");
      if(isCreateSchema(config)) return;

      for(auto &antennaList : antennaLists)
        for(auto &antenna : antennaList->antennas)
          for(auto &pattern : antenna->patterns)
            if(!types.size() || pattern.type.isInList(types))
            {
              if(zeroOffset)  pattern.offset = Vector3d(0., 0., 0.);
              if(zeroPattern) pattern.pattern.setNull();
            }

      for(auto &antennaList : antennaLists)
        antennas.insert(antennas.end(), antennaList->antennas.begin(), antennaList->antennas.end());
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

static const char *docstringGnssAntennaDefintionListRemoveCenterMean = R"(
\subsection{RemoveCenterMean}
The antenna offset and antenna variations (patterns) are inseparable parts of the
antenna model. With \config{removeOffset} an estimated offset is removed from
all selected patterns and added to the offset. With \config{removeMean} an estimated
constant is removed additionally as it cannot be seperated from signal biases.
The mean and offset are defined as discretized (\config{deltaAzimuth},
\config{dZenith}) integral of the spherical cap from zenith down to \config{maxZenith}.
)";

class GnssAntennaDefintionListRemoveCenterMean : public GnssAntennaDefintionList
{
public:
  GnssAntennaDefintionListRemoveCenterMean(Config &config)
  {
    try
    {
      std::vector<GnssAntennaDefintionListPtr> antennaLists;
      std::vector<GnssType> types;
      Bool  removeOffset, removeMean;
      Angle dAzimuth, dZenith, maxZenith;

      readConfig(config, "antenna",      antennaLists, Config::MUSTSET, "",   "");
      readConfig(config, "patternTypes", types,        Config::OPTIONAL, "",  "only matching patterns, default: all");
      readConfig(config, "removeMean",   removeMean,   Config::MUSTSET, "1",  "");
      readConfig(config, "removeOffset", removeOffset, Config::MUSTSET, "1",  "");
      readConfig(config, "deltaAzimuth", dAzimuth,     Config::DEFAULT, "1",  "[degree] sampling of pattern to estimate center/constant");
      readConfig(config, "deltaZenith",  dZenith,      Config::DEFAULT, "1",  "[degree] sampling of pattern to estimate center/constant");
      readConfig(config, "maxZenith",    maxZenith,    Config::DEFAULT, "90", "[degree] sampling of pattern to estimate center/constant");
      if(isCreateSchema(config)) return;

      const UInt rows = static_cast<UInt>(std::round(2*PI/dAzimuth));
      const UInt cols = static_cast<UInt>(std::round(maxZenith/dZenith));
      Matrix A(rows*cols, 3*removeOffset+removeMean);
      UInt idx = 0;
      for(UInt i=0; i<rows; i++)
        for(UInt k=0; k<cols; k++)
        {
          const Angle azimuth  (i*2*PI/rows);
          const Angle elevation(PI/2-(k+0.5)*Double(dZenith));
          if(removeOffset) A(idx, 0) = -std::cos(elevation) * std::cos(azimuth); // x
          if(removeOffset) A(idx, 1) = -std::cos(elevation) * std::sin(azimuth); // y
          if(removeOffset) A(idx, 2) = -std::sin(elevation);                     // z
          if(removeMean)   A(idx, 3*removeOffset) = 1;                           // mean
          A.row(idx) *= std::sqrt(std::cos(elevation));                          // area weights
          idx++;
        }

      Vector l(rows*cols);
      for(auto &antennaList : antennaLists)
        for(auto &antenna : antennaList->antennas)
          for(auto &pattern : antenna->patterns)
            if(!types.size() || pattern.type.isInList(types))
            {
              Matrix PA = A;
              UInt idx = 0;
              for(UInt i=0; i<rows; i++)
                for(UInt k=0; k<cols; k++)
                {
                  const Angle azimuth  (i*2*PI/rows);
                  const Angle elevation(PI/2-(k+0.5)*Double(dZenith));
                  l(idx) = pattern.antennaVariations(azimuth, elevation, FALSE/*applyOffset*/) * std::sqrt(std::cos(elevation));
                  if(std::isnan(l(idx)))
                  {
                    l(idx) = 0.;
                    PA.row(idx).setNull();
                  }
                  idx++;
                }
              Vector x = leastSquares(PA, l);
              if(removeOffset)
                pattern.offset += Vector3d(x(0), x(1), x(2));
              // remove from pattern
              for(UInt i=0; i<pattern.pattern.rows(); i++)
                for(UInt k=0; k<pattern.pattern.columns(); k++)
                {
                  if(removeOffset)
                  {
                    const Angle azimuth  (i*2*PI/pattern.pattern.rows());
                    const Angle elevation(PI/2-k*Double(pattern.dZenit));
                    pattern.pattern(i, k) += std::cos(elevation) * std::cos(azimuth) * x(0)
                                           + std::cos(elevation) * std::sin(azimuth) * x(1)
                                           + std::sin(elevation)                     * x(2);
                  }
                  if(removeMean)
                    pattern.pattern(i, k) -= x(3*removeOffset);
                }
            } // for(pattern)

      for(auto &antennaList : antennaLists)
        antennas.insert(antennas.end(), antennaList->antennas.begin(), antennaList->antennas.end());
    }
    catch(std::exception &e)
    {
      GROOPS_RETHROW(e)
    }
  }
};

/***********************************************/

GROOPS_REGISTER_CLASS(GnssAntennaDefintionList, "gnssAntennaDefintionListType",
                      GnssAntennaDefintionListNew,
                      GnssAntennaDefintionListFromFile,
                      GnssAntennaDefintionListFromStationInfo,
                      GnssAntennaDefintionListResample,
                      GnssAntennaDefintionListTransform,
                      GnssAntennaDefintionListRename,
                      GnssAntennaDefintionListSetZero,
                      GnssAntennaDefintionListRemoveCenterMean)

GROOPS_READCONFIG_CLASS(GnssAntennaDefintionList, "gnssAntennaDefintionListType")

/***********************************************/

GnssAntennaDefintionListPtr GnssAntennaDefintionList::create(Config &config, const std::string &name)
{
  try
  {
    GnssAntennaDefintionListPtr ptr;
    std::string           choice;
    readConfigChoice(config, name, choice, Config::MUSTSET, "", "");
    if(readConfigChoiceElement(config, "new",              choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListNew(config));
    if(readConfigChoiceElement(config, "fromFile",         choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListFromFile(config));
    if(readConfigChoiceElement(config, "fromStationInfo",  choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListFromStationInfo(config));
    if(readConfigChoiceElement(config, "resample",         choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListResample(config));
    if(readConfigChoiceElement(config, "transform",        choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListTransform(config));
    if(readConfigChoiceElement(config, "rename",           choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListRename(config));
    if(readConfigChoiceElement(config, "setZero",          choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListSetZero(config));
    if(readConfigChoiceElement(config, "removeCenterMean", choice, ""))
      ptr = GnssAntennaDefintionListPtr(new GnssAntennaDefintionListRemoveCenterMean(config));
    endChoice(config);
    return ptr;
  }
  catch(std::exception &e)
  {
    GROOPS_RETHROW(e)
  }
}

/***********************************************/

void GnssAntennaDefinitionCreate::run(Config &config, Parallel::CommunicatorPtr /*comm*/)
{
  try
  {
    FileName fileNameOutAntenna;
    std::vector<GnssAntennaDefintionListPtr> antennaLists;

    readConfig(config, "outputfileAntennaDefinition", fileNameOutAntenna, Config::MUSTSET, "", "");
    readConfig(config, "antenna",                     antennaLists,       Config::MUSTSET, "", "");
    if(isCreateSchema(config)) return;

    // ==================================

    std::vector<GnssAntennaDefinitionPtr> antennas;
    for(const auto &antennaList : antennaLists)
      antennas.insert(antennas.end(), antennaList->antennas.begin(), antennaList->antennas.end());

    std::stable_sort(antennas.begin(), antennas.end(), [](GnssAntennaDefinitionPtr a, GnssAntennaDefinitionPtr b)
                     {return (a->name != b->name) ? (a->name < b->name) : ((a->serial != b->serial) ? (a->serial < b->serial) : (a->radome < b->radome));});
    antennas.erase(std::unique(antennas.begin(), antennas.end(), [](GnssAntennaDefinitionPtr a, GnssAntennaDefinitionPtr b) {return (a->name == b->name) && (a->serial == b->serial) && (a->radome == b->radome);}), antennas.end());


    logStatus<<"write antenna definition <"<<fileNameOutAntenna<<">"<<Log::endl;
    writeFileGnssAntennaDefinition(fileNameOutAntenna, antennas);

    for(const auto &antenna : antennas)
    {
      logInfo<<" "<<antenna->str()<<" "<<antenna->comment<<Log::endl;
      for(const auto &pattern : antenna->patterns)
        logInfo<<"   "<<pattern.type.str()<<" offset("<<pattern.offset.x()*1e3%"%5.1f, "s<<pattern.offset.y()*1e3%"%5.1f, "s<<pattern.offset.z()*1e3%"%6.1f) mm,"s
               <<" pattern("s<<360./pattern.pattern.rows()%"%4.1f° x "s<<RAD2DEG*pattern.dZenit%"%3.1f° ),"s
               <<" maxZenith "s<<(pattern.pattern.columns()-1)*RAD2DEG*pattern.dZenit%"%4.1f°"s<<Log::endl;
    }
  }
  catch(std::exception &e)
  {
    GROOPS_RETHROW(e)
  }
}

/***********************************************/
