/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::common::f_fmla;
use crate::polyeval::f_polyeval4;
use crate::sin_cosf::sincosf_eval::sincosf_eval;

/// Computes cos(x) - 1
///
/// ULP 0.5
pub fn f_cosm1f(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let x = f32::from_bits(x_abs);

    // |x| <= pi/16
    if x_abs <= 0x3e49_0fdbu32 {
        let xd = x as f64;
        // |x| < 0.000244141
        if x_abs < 0x3980_0000u32 {
            // Taylor expansion for small cos(x) - 1 ~ -x^2/2 + x^4/24 + O(x^6)
            let x_sqr = xd * xd;
            const R: f64 = 1. / 2.;
            return (-x_sqr * R) as f32;
        }

        // Cosine
        // Generated poly by Sollya:
        // Polynomial cos(x) - 1 = x^2 * P(x^2)
        //
        // d = [0.0000000000001, pi/16];
        // f_cosm1 = cos(x) - 1;
        // Q = fpminimax(f_cosm1, [|2,4,6,8|], [|0, D...|], d);

        let x2 = xd * xd;
        let p = f_polyeval4(
            x2,
            f64::from_bits(0xbfe0000000000000),
            f64::from_bits(0x3fa55555554ed21a),
            f64::from_bits(0xbf56c16b981a61d0),
            f64::from_bits(0x3ef9fc205e761f45),
        );
        return (p * x2) as f32;
    }

    if x_abs >= 0x7f80_0000u32 {
        return x + f32::NAN;
    }

    // cos(x) - 1 = -2*sin^2(x/2)
    // Hence we're computing sin using formula:
    //   sin(x) = sin((k + y)*pi/32)
    //          = sin(y*pi/32) * cos(k*pi/32) + cos(y*pi/32) * sin(k*pi/32)
    // The values of sin(k*pi/32) and cos(k*pi/32) for k = 0..31 are precomputed
    // and stored using a vector of 32 doubles. Sin(y*pi/32) and cos(y*pi/32) are
    // computed using degree-7 and degree-6 minimax polynomials generated by
    // Sollya respectively.

    let x_abs = (x * 0.5).to_bits();
    let xd = f32::from_bits(x_abs) as f64;

    let rs = sincosf_eval(xd, x_abs);
    let sin_x_over_2 = f_fmla(rs.sin_y, rs.cos_k, f_fmla(rs.cosm1_y, rs.sin_k, rs.sin_k));
    let sin_sqr = sin_x_over_2 * sin_x_over_2;
    let cosm1x = -2. * sin_sqr;
    cosm1x as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn f_cosm1f_test() {
        assert_eq!(f_cosm1f(0.00015928394), -1.2685687e-8);
        assert_eq!(f_cosm1f(0.0), 0.0);
        assert_eq!(f_cosm1f(std::f32::consts::PI), -2.);
        assert_eq!(f_cosm1f(0.5), -0.122417435);
        assert_eq!(f_cosm1f(0.7), -0.2351578);
        assert_eq!(f_cosm1f(1.7), -1.1288445);
        assert!(f_cosm1f(f32::INFINITY).is_nan());
        assert!(f_cosm1f(f32::NEG_INFINITY).is_nan());
        assert!(f_cosm1f(f32::NAN).is_nan());
        assert_eq!(f_cosm1f(0.0002480338), -3.076038e-8);
    }
}
