/*
 * Copyright (C) 2024 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "AXCoreObject.h"
#include <wtf/HashMap.h>
#include <wtf/Vector.h>
#include <wtf/WeakPtr.h>

namespace WebCore {

DECLARE_ALLOCATOR_WITH_HEAP_IDENTIFIER(AXSearchManager);

enum class AccessibilitySearchKey {
    AnyType = 1,
    Article,
    BlockquoteSameLevel,
    Blockquote,
    BoldFont,
    Button,
    Checkbox,
    Control,
    DifferentType,
    FontChange,
    FontColorChange,
    Frame,
    Graphic,
    HeadingLevel1,
    HeadingLevel2,
    HeadingLevel3,
    HeadingLevel4,
    HeadingLevel5,
    HeadingLevel6,
    HeadingSameLevel,
    Heading,
    Highlighted,
    ItalicFont,
    KeyboardFocusable,
    Landmark,
    Link,
    List,
    LiveRegion,
    MisspelledWord,
    Outline,
    PlainText,
    RadioGroup,
    SameType,
    StaticText,
    StyleChange,
    TableSameLevel,
    Table,
    TextField,
    Underline,
    UnvisitedLink,
    VisitedLink,
};

struct AccessibilitySearchCriteria {
    // FIXME: change the object pointers to object IDs.
    WeakPtr<AXCoreObject> anchorObject { nullptr };
    WeakPtr<AXCoreObject> startObject { nullptr };
    CharacterRange startRange;
    AccessibilitySearchDirection searchDirection { AccessibilitySearchDirection::Next };
    Vector<AccessibilitySearchKey> searchKeys;
    String searchText;
    unsigned resultsLimit { 0 };
    bool visibleOnly { false };
    bool immediateDescendantsOnly { false };
};

class AXSearchManager {
    WTF_DEPRECATED_MAKE_FAST_ALLOCATED_WITH_HEAP_IDENTIFIER(AXSearchManager, AXSearchManager);
public:
    AXCoreObject::AccessibilityChildrenVector findMatchingObjects(AccessibilitySearchCriteria&&);
    std::optional<AXTextMarkerRange> findMatchingRange(AccessibilitySearchCriteria&&);
private:
    AXCoreObject::AccessibilityChildrenVector findMatchingObjectsInternal(const AccessibilitySearchCriteria&);
    bool matchWithResultsLimit(Ref<AXCoreObject>, const AccessibilitySearchCriteria&, AXCoreObject::AccessibilityChildrenVector&);
    bool match(Ref<AXCoreObject>, const AccessibilitySearchCriteria&);
    bool matchText(Ref<AXCoreObject>, const String&);
    bool matchForSearchKeyAtIndex(Ref<AXCoreObject>, const AccessibilitySearchCriteria&, size_t);
    DidTimeout revealHiddenMatchWithTimeout(AXCoreObject&, Seconds);

    bool lastRevealAttemptTimedOut()
    {
        if (isMainThread())
            return false;
        return m_lastRevealAttemptTimedOut;
    }
    void setLastRevealAttemptTimedOut(bool newValue)
    {
        AX_ASSERT(!isMainThread());
        m_lastRevealAttemptTimedOut = newValue;
    }

    // Keeps the ranges of misspellings for each object.
    HashMap<AXID, Vector<AXTextMarkerRange>> m_misspellingRanges;

    // For certain types of searches, we may detect that an object matching the search is in a collapsed,
    // but revealable / expandable container. We try to do this reveal synchronously from the accessibility thread
    // to the main-thread, but with a timeout in case the main-thread is busy. If the main-thread is busy once,
    // we don't want to try to synchronously reveal collapsed content again.
    //
    // This must only be read and written from the accessibility thread.
    bool m_lastRevealAttemptTimedOut { false };
};

inline AXCoreObject::AccessibilityChildrenVector AXSearchManager::findMatchingObjects(AccessibilitySearchCriteria&& criteria)
{
    return findMatchingObjectsInternal(std::forward<AccessibilitySearchCriteria>(criteria));
}

WTF::TextStream& operator<<(WTF::TextStream&, AccessibilitySearchKey);
WTF::TextStream& operator<<(WTF::TextStream&, const AccessibilitySearchCriteria&);

} // namespace WebCore
