// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

#ifdef FEATURE_INTERPRETER

#include "threads.h"
#include "gcenv.h"
#include "interpexec.h"
#include "frames.h"

// for numeric_limits
#include <limits>

#ifdef TARGET_WASM
void InvokeCalliStub(PCODE ftn, CallStubHeader *stubHeaderTemplate, int8_t *pArgs, int8_t *pRet);
void InvokeCompiledMethod(MethodDesc *pMD, int8_t *pArgs, int8_t *pRet, PCODE target);
void InvokeDelegateInvokeMethod(MethodDesc *pMDDelegateInvoke, int8_t *pArgs, int8_t *pRet, PCODE target);
#else
#include "callstubgenerator.h"

CallStubHeader *UpdateCallStubForMethod(MethodDesc *pMD)
{
    CONTRACTL
    {
        THROWS;
        MODE_ANY;
        PRECONDITION(CheckPointer(pMD));
    }
    CONTRACTL_END

    GCX_PREEMP();

    CallStubGenerator callStubGenerator;

    AllocMemTracker amTracker;
    CallStubHeader *header = callStubGenerator.GenerateCallStub(pMD, &amTracker, true /* interpreterToNative */);

    if (pMD->SetCallStub(header))
    {
        amTracker.SuppressRelease();
    }
    else
    {
        // We have lost the race for generating the header, use the one that was generated by another thread
        // and let the amTracker release the memory of the one we generated.
        header = pMD->GetCallStub();
    }

    return header;
}

void InvokeCompiledMethod(MethodDesc *pMD, int8_t *pArgs, int8_t *pRet, PCODE target)
{
    CONTRACTL
    {
        THROWS;
        MODE_ANY;
        PRECONDITION(CheckPointer(pMD));
        PRECONDITION(CheckPointer(pArgs));
        PRECONDITION(CheckPointer(pRet));
    }
    CONTRACTL_END

    CallStubHeader *pHeader = pMD->GetCallStub();
    if (pHeader == NULL)
    {
        pHeader = UpdateCallStubForMethod(pMD);
    }

    // Interpreter-FIXME: Potential race condition if a single CallStubHeader is reused for multiple targets.
    pHeader->SetTarget(target); // The method to call

    pHeader->Invoke(pHeader->Routines, pArgs, pRet, pHeader->TotalStackSize);
}

void InvokeDelegateInvokeMethod(MethodDesc *pMDDelegateInvoke, int8_t *pArgs, int8_t *pRet, PCODE target)
{
    CONTRACTL
    {
        THROWS;
        MODE_ANY;
        PRECONDITION(CheckPointer(pMDDelegateInvoke));
        PRECONDITION(CheckPointer(pArgs));
        PRECONDITION(CheckPointer(pRet));
    }
    CONTRACTL_END

    CallStubHeader *stubHeaderTemplate = pMDDelegateInvoke->GetCallStub();
    if (stubHeaderTemplate == NULL)
    {
        stubHeaderTemplate = UpdateCallStubForMethod(pMDDelegateInvoke);
    }

    // CallStubHeaders encode their destination addresses in the Routines array, so they need to be
    // copied to a local buffer before we can actually set their target address.
    size_t templateSize = stubHeaderTemplate->GetSize();
    uint8_t* actualCallStub = (uint8_t*)alloca(templateSize);
    memcpy(actualCallStub, stubHeaderTemplate, templateSize);
    CallStubHeader *pHeader = (CallStubHeader*)actualCallStub;
    pHeader->SetTarget(target); // The method to call
    pHeader->Invoke(pHeader->Routines, pArgs, pRet, pHeader->TotalStackSize);
}

void InvokeCalliStub(PCODE ftn, CallStubHeader *stubHeaderTemplate, int8_t *pArgs, int8_t *pRet)
{
    CONTRACTL
    {
        THROWS;
        MODE_ANY;
        PRECONDITION(CheckPointer((void*)ftn));
        PRECONDITION(CheckPointer(stubHeaderTemplate));
    }
    CONTRACTL_END

    // CallStubHeaders encode their destination addresses in the Routines array, so they need to be
    // copied to a local buffer before we can actually set their target address.
    size_t templateSize = stubHeaderTemplate->GetSize();
    uint8_t* actualCallStub = (uint8_t*)alloca(templateSize);
    memcpy(actualCallStub, stubHeaderTemplate, templateSize);
    CallStubHeader *pHeader = (CallStubHeader*)actualCallStub;
    pHeader->SetTarget(ftn); // The method to call
    pHeader->Invoke(pHeader->Routines, pArgs, pRet, pHeader->TotalStackSize);
}

// Create call stub for calling interpreted methods from JITted/AOTed code.
CallStubHeader *CreateNativeToInterpreterCallStub(InterpMethod* pInterpMethod)
{
    CallStubGenerator callStubGenerator;
    CallStubHeader *pHeader = VolatileLoadWithoutBarrier(&pInterpMethod->pCallStub);
    GCX_PREEMP();

    if (pHeader == NULL)
    {
        // Ensure that there is an interpreter thread context instance and thus an interpreter stack
        // allocated for this thread. This allows us to not to have to check and allocate it from
        // the interpreter stub right after this call.
        GetThread()->GetInterpThreadContext();

        GCX_PREEMP();

        AllocMemTracker amTracker;
        pHeader = callStubGenerator.GenerateCallStub((MethodDesc*)pInterpMethod->methodHnd, &amTracker, false /* interpreterToNative */);

        if (InterlockedCompareExchangeT(&pInterpMethod->pCallStub, pHeader, NULL) == NULL)
        {
            amTracker.SuppressRelease();
        }
        else
        {
            // We have lost the race for generating the header, use the one that was generated by another thread
            // and let the amTracker release the memory of the one we generated.
            pHeader = VolatileLoadWithoutBarrier(&pInterpMethod->pCallStub);
        }
    }

    return pHeader;
}

#endif // !TARGET_WASM

typedef void* (*HELPER_FTN_P_P)(void*);
typedef void* (*HELPER_FTN_BOX_UNBOX)(MethodTable*, void*);
typedef Object* (*HELPER_FTN_NEWARR)(MethodTable*, intptr_t);
typedef void* (*HELPER_FTN_P_PP)(void*, void*);
typedef void (*HELPER_FTN_V_PPP)(void*, void*, void*);


InterpThreadContext::InterpThreadContext()
{
    // FIXME VirtualAlloc/mmap with INTERP_STACK_ALIGNMENT alignment
    pStackStart = pStackPointer = (int8_t*)malloc(INTERP_STACK_SIZE);
    pStackEnd = pStackStart + INTERP_STACK_SIZE;
}

InterpThreadContext::~InterpThreadContext()
{
    free(pStackStart);
}

DictionaryEntry GenericHandleWorkerCore(MethodDesc * pMD, MethodTable * pMT, LPVOID signature, DWORD dictionaryIndexAndSlot, Module* pModule);

void* GenericHandleCommon(MethodDesc * pMD, MethodTable * pMT, LPVOID signature)
{
    CONTRACTL
    {
        THROWS;
        GC_TRIGGERS;
        MODE_COOPERATIVE;
    } CONTRACTL_END;
    GCX_PREEMP();
    return GenericHandleWorkerCore(pMD, pMT, signature, 0xFFFFFFFF, NULL);
}

#ifdef DEBUG
static void InterpBreakpoint()
{

}
#endif

static OBJECTREF CreateMultiDimArray(MethodTable* arrayClass, int8_t* stack, int32_t dimsOffset, int numArgs)
{
    int32_t* dims = (int32_t*)alloca(numArgs * sizeof(int32_t));
    for (int i = 0; i < numArgs; i++)
    {
        dims[i] = *(int32_t*)(stack + dimsOffset + i * 8);
    }

    return AllocateArrayEx(arrayClass, dims, numArgs);
}

#define LOCAL_VAR_ADDR(offset,type) ((type*)(stack + (offset)))
#define LOCAL_VAR(offset,type) (*LOCAL_VAR_ADDR(offset, type))
#define NULL_CHECK(o) do { if ((o) == NULL) { COMPlusThrow(kNullReferenceException); } } while (0)

template <typename THelper> static THelper GetPossiblyIndirectHelper(const InterpMethod *pMethod, int32_t _data)
{
    InterpHelperData data;
    memcpy(&data, &_data, sizeof(int32_t));

    void *addr = pMethod->pDataItems[data.addressDataItemIndex];
    switch (data.accessType) {
        case IAT_VALUE:
            return (THelper)addr;
        case IAT_PVALUE:
            return *(THelper *)addr;
        case IAT_PPVALUE:
            return **(THelper **)addr;
        default:
            COMPlusThrowHR(COR_E_EXECUTIONENGINE);
            return (THelper)nullptr;
    }
}

// At present our behavior for float to int conversions is to perform a saturating conversion down to either 32 or 64 bits
//  and then perform an unchecked truncation from that intermediate size down to the actual result size.
// See https://github.com/dotnet/runtime/issues/116823
template <typename TResult, typename TIntermediate, typename TSource> static void ConvFpHelper(int8_t *stack, const int32_t *ip)
{
    static_assert(!std::numeric_limits<TSource>::is_integer, "ConvFpHelper is only for use on floats and doubles");
    static_assert(sizeof(TIntermediate) >= sizeof(TResult), "Intermediate type must not be smaller than result type");
    static_assert(std::numeric_limits<TResult>::is_integer, "ConvFpHelper is only for use on floats and doubles to be converted to integers");

    // First, promote the source value to double
    double src = LOCAL_VAR(ip[2], TSource),
        minValue = (double)std::numeric_limits<TIntermediate>::lowest(),
        maxValue = (double)std::numeric_limits<TIntermediate>::max();

    // (src != src) checks for NaN, then we check whether the min and max values (as represented by their closest double)
    //  properly bound the source value so that when it is truncated it will be in range
    // We assume that we are in round-towards-zero mode. For NaN we want to return 0, and for out of range values, saturate.
    TResult result;
    if (src != src)
        result = 0;
    else if (src >= maxValue)
        result = (TResult)std::numeric_limits<TIntermediate>::max();
    else if (!std::numeric_limits<TIntermediate>::is_signed && (src <= -1))
        result = 0;
    else if (std::numeric_limits<TIntermediate>::is_signed && (src < minValue))
        result = (TResult)std::numeric_limits<TIntermediate>::lowest();
    else
        result = (TResult)(TIntermediate)src;

    // According to spec, for result types smaller than int32, we store them on the stack as int32
    if (sizeof(TResult) >= 4)
        LOCAL_VAR(ip[1], TResult) = result;
    else
        LOCAL_VAR(ip[1], int32_t) = (int32_t)result;
}

template <typename TResult, typename TSource> static void ConvOvfFpHelper(int8_t *stack, const int32_t *ip)
{
    static_assert(!std::numeric_limits<TSource>::is_integer, "ConvOvfFpHelper is only for use on floats and doubles");
    static_assert(std::numeric_limits<TResult>::is_integer, "ConvOvfFpHelper is only for use on floats and doubles to be converted to integers");
    static_assert(sizeof(TResult) <= 4, "ConvOvfFpHelper's generic version is only for use on results <= 4 bytes in size");

    // First, promote the source value to double
    double src = LOCAL_VAR(ip[2], TSource),
        minValue = (double)std::numeric_limits<TResult>::lowest() - 1,
        maxValue = (double)std::numeric_limits<TResult>::max() + 1;

    // We assume that we are in round-towards-zero mode
    bool inRange = (src > minValue) && (src < maxValue);

    if (!inRange)
        COMPlusThrow(kOverflowException);

    TResult truncated = (TResult)src;
    // According to spec, for result types smaller than int32, we store them on the stack as int32
    LOCAL_VAR(ip[1], int32_t) = (int32_t)truncated;
}

// I64 and U64 versions based on mono_math.h

template <typename TSource> static void ConvOvfFpHelperI64(int8_t *stack, const int32_t *ip)
{
    static_assert(!std::numeric_limits<TSource>::is_integer, "ConvOvfFpHelper is only for use on floats and doubles");

    const double two63 = 2147483648.0 * 4294967296.0;
    // First, promote the source value to double
    double src = LOCAL_VAR(ip[2], TSource),
        // Define the boundary values we need to be between (see System.Math.ConvertToInt64Checked)
        minValue = (-two63 - 0x402),
        maxValue = two63;

    bool inRange = (src > minValue) && (src < maxValue);
    if (!inRange)
        COMPlusThrow(kOverflowException);

    int64_t truncated = (int64_t)src;
    LOCAL_VAR(ip[1], int64_t) = truncated;
}

template <typename TSource> static void ConvOvfFpHelperU64(int8_t *stack, const int32_t *ip)
{
    static_assert(!std::numeric_limits<TSource>::is_integer, "ConvOvfFpHelper is only for use on floats and doubles");

    // First, promote the source value to double
    double src = LOCAL_VAR(ip[2], TSource),
        // Define the boundary values we need to be between (see System.Math.ConvertToUInt64Checked)
        minValue = -1.0,
        maxValue = 4294967296.0 * 4294967296.0;

    bool inRange = (src > minValue) && (src < maxValue);
    if (!inRange)
        COMPlusThrow(kOverflowException);

    uint64_t truncated = (uint64_t)src;
    LOCAL_VAR(ip[1], uint64_t) = truncated;
}

template <typename TResult, typename TSource> void ConvOvfHelper(int8_t *stack, const int32_t *ip)
{
    static_assert(std::numeric_limits<TSource>::is_integer, "ConvOvfHelper is only for use on integers");
    constexpr bool shrinking = sizeof(TResult) < sizeof(TSource);

    TSource src = LOCAL_VAR(ip[2], TSource);
    bool inRange;
    if (shrinking)
    {
        if (std::numeric_limits<TSource>::is_signed)
            inRange = (src >= (TSource)std::numeric_limits<TResult>::lowest()) && (src <= (TSource)std::numeric_limits<TResult>::max());
        else
            inRange = (src <= (TSource)std::numeric_limits<TResult>::max());
    }
    else
    {
        // Growing conversions with the same signedness shouldn't use an ovf opcode.
        static_assert(
            shrinking || (std::numeric_limits<TResult>::is_signed != std::numeric_limits<TSource>::is_signed),
            "ConvOvfHelper only does growing conversions with sign changes"
        );

        if (std::numeric_limits<TResult>::is_signed)
        {
            if (sizeof(TSource) == sizeof(TResult))
            {
                // unsigned -> signed conv with same size. check to make sure the source value isn't too big.
                inRange = src <= (TSource)std::numeric_limits<TResult>::max();
            }
            else
            {
                // growing unsigned -> signed conversion. this can never fail.
                inRange = true;
            }
        }
        else
        {
            // signed -> unsigned conv. check to make sure the source value isn't negative.
            inRange = src >= 0;
        }
    }

    if (inRange)
    {
        // conv_ovf with floating point inputs is handled in ConvOvfFpHelper, so all possible conv_ovfs in this function are
        // from int64 into int32-or-smaller, or int32-or-smaller into int16-or-smaller.
        // The spec says that conversion results are stored on the evaluation stack as signed, so we always want to convert
        //  the final truncated result into int32_t before storing it on the stack, even if the truncated result is unsigned.
        TResult result = (TResult)src;
        if (sizeof (TResult) < 4)
            LOCAL_VAR(ip[1], int32_t) = (int32_t)result;
        else
            LOCAL_VAR(ip[1], TResult) = result;
    }
    else
    {
        COMPlusThrow(kOverflowException);
    }
}

void* DoGenericLookup(void* genericVarAsPtr, InterpGenericLookup* pLookup)
{
    // TODO! If this becomes a performance bottleneck, we could expand out the various permutations of this
    // so that we have 24 versions of lookup (or 48 is we allow for avoiding the null check), do the only
    // if check to figure out which one to use, and then have the rest of the logic be straight-line code.
    MethodTable *pMT = nullptr;
    MethodDesc* pMD = nullptr;
    uint8_t* lookup;
    if (pLookup->lookupType == InterpGenericLookupType::This)
    {
        OBJECTREF thisPtr = ObjectToOBJECTREF((Object*)genericVarAsPtr);
        NULL_CHECK(thisPtr);
        pMT = thisPtr->GetMethodTable();
        lookup = (uint8_t*)pMT;
    }
    else if (pLookup->lookupType == InterpGenericLookupType::Class)
    {
        pMT = (MethodTable*)genericVarAsPtr;
        lookup = (uint8_t*)pMT;
    }
    else
    {
        assert(pLookup->lookupType == InterpGenericLookupType::Method);
        pMD = (MethodDesc*)genericVarAsPtr;
        lookup = (uint8_t*)pMD;
    }
    void* result = 0;

    uint16_t indirections = pLookup->indirections;
    if (indirections == 0)
    {
        return lookup;
    }
    else if (indirections != InterpGenericLookup_UseHelper)
    {
        lookup = VolatileLoadWithoutBarrier((uint8_t**)(lookup + pLookup->offsets[0]));
        if (indirections >= 3)
            lookup = VolatileLoadWithoutBarrier((uint8_t**)(lookup + pLookup->offsets[1]));
        if (indirections >= 4)
            lookup = VolatileLoadWithoutBarrier((uint8_t**)(lookup + pLookup->offsets[2]));
        do {
            size_t lastOffset = pLookup->offsets[indirections - 1];
            if (pLookup->sizeOffset != CORINFO_NO_SIZE_CHECK)
            {
                /* Last indirection to check is the size*/
                size_t size = VolatileLoadWithoutBarrier((size_t*)(lookup + pLookup->sizeOffset));
                if (size <= lastOffset)
                {
                    result = GenericHandleCommon(pMD, pMT, pLookup->signature);
                    break;
                }
            }
            lookup = VolatileLoadWithoutBarrier((uint8_t**)(lookup + lastOffset));

            if (lookup == NULL)
            {
                result = GenericHandleCommon(pMD, pMT, pLookup->signature);
                break;
            }
            result = lookup;
        } while (0);
    }
    else
    {
        return GenericHandleCommon(pMD, pMT, pLookup->signature);
    }
    return result;
}

void InterpExecMethod(InterpreterFrame *pInterpreterFrame, InterpMethodContextFrame *pFrame, InterpThreadContext *pThreadContext, ExceptionClauseArgs *pExceptionClauseArgs)
{
    CONTRACTL
    {
        GC_TRIGGERS;
        MODE_COOPERATIVE;
    }
    CONTRACTL_END;

#if defined(HOST_AMD64) && defined(HOST_WINDOWS)
    pInterpreterFrame->SetInterpExecMethodSSP((TADDR)_rdsspq());
#endif // HOST_AMD64 && HOST_WINDOWS

    const int32_t *ip;
    int8_t *stack;

    InterpMethod *pMethod = pFrame->startIp->Method;
    assert(pMethod->CheckIntegrity());

    pThreadContext->pStackPointer = pFrame->pStack + pMethod->allocaSize;
    stack = pFrame->pStack;

    if (pExceptionClauseArgs == NULL)
    {
        // Start executing at the beginning of the method
        ip = pFrame->startIp->GetByteCodes();
    }
    else
    {
        // * Filter funclets are executed in the current frame, because they are executed
        //   in the first pass of EH when the frames between the current frame and the
        //   parent frame are still alive. All accesses to the locals and arguments
        //   in this case use the pExceptionClauseArgs->pFrame->pStack as a frame pointer.
        // * Catch and finally funclets are running in the parent frame directly

        if (pExceptionClauseArgs->isFilter)
        {
            // Since filters run in their own frame, we need to clear the global variables
            // so that GC doesn't pick garbage in variables that were not yet written to.
            memset(pFrame->pStack, 0, pMethod->allocaSize);
        }

        // Start executing at the beginning of the exception clause
        ip = pExceptionClauseArgs->ip;
    }

    int32_t returnOffset, callArgsOffset, methodSlot;
    MethodDesc* targetMethod;

MAIN_LOOP:
    try
    {
        INSTALL_MANAGED_EXCEPTION_DISPATCHER;
        INSTALL_UNWIND_AND_CONTINUE_HANDLER;
        while (true)
        {
            // Interpreter-TODO: This is only needed to enable SOS see the exact location in the interpreted method.
            // Neither the GC nor the managed debugger needs that as they walk the stack when the runtime is suspended
            // and we can save the IP to the frame at the suspension time.
            // It will be useful for testing e.g. the debug info at various locations in the current method, so let's
            // keep it for such purposes until we don't need it anymore.
            pFrame->ip = (int32_t*)ip;

            switch (*ip)
            {
#ifdef DEBUG
                case INTOP_BREAKPOINT:
                    InterpBreakpoint();
                    ip++;
                    break;
#endif
                case INTOP_INITLOCALS:
                    memset(stack + ip[1], 0, ip[2]);
                    ip += 3;
                    break;
                case INTOP_MEMBAR:
                    MemoryBarrier();
                    ip++;
                    break;
                case INTOP_LDC_I4:
                    LOCAL_VAR(ip[1], int32_t) = ip[2];
                    ip += 3;
                    break;
                case INTOP_LDC_I4_0:
                    LOCAL_VAR(ip[1], int32_t) = 0;
                    ip += 2;
                    break;
                case INTOP_LDC_I8_0:
                    LOCAL_VAR(ip[1], int64_t) = 0;
                    ip += 2;
                    break;
                case INTOP_LDC_I8:
                    LOCAL_VAR(ip[1], int64_t) = (int64_t)(uint32_t)ip[2] + ((int64_t)ip[3] << 32);
                    ip += 4;
                    break;
                case INTOP_LDC_R4:
                    LOCAL_VAR(ip[1], int32_t) = ip[2];
                    ip += 3;
                    break;
                case INTOP_LDC_R8:
                    LOCAL_VAR(ip[1], int64_t) = (int64_t)(uint32_t)ip[2] + ((int64_t)ip[3] << 32);
                    ip += 4;
                    break;
                case INTOP_LDPTR:
                    LOCAL_VAR(ip[1], void*) = pMethod->pDataItems[ip[2]];
                    ip += 3;
                    break;
                case INTOP_LDPTR_DEREF:
                    LOCAL_VAR(ip[1], void*) = *(void**)pMethod->pDataItems[ip[2]];
                    ip += 3;
                    break;
                case INTOP_NULLCHECK:
                    NULL_CHECK(LOCAL_VAR(ip[1], void*));
                    ip += 2;
                    break;
                case INTOP_RET:
                    // Return stack slot sized value
                    *(int64_t*)pFrame->pRetVal = LOCAL_VAR(ip[1], int64_t);
                    goto EXIT_FRAME;
                case INTOP_RET_VT:
                    memmove(pFrame->pRetVal, stack + ip[1], ip[2]);
                    goto EXIT_FRAME;
                case INTOP_RET_VOID:
                    goto EXIT_FRAME;

                case INTOP_LDLOCA:
                    LOCAL_VAR(ip[1], void*) = stack + ip[2];
                    ip += 3;
                    break;
                case INTOP_LOAD_FRAMEVAR:
                    _ASSERTE((pExceptionClauseArgs != NULL) && (pExceptionClauseArgs->isFilter));
                    LOCAL_VAR(ip[1], void*) = pExceptionClauseArgs->pFrame->pStack;
                    ip += 2;
                    break;

#define MOV(argtype1,argtype2) \
    LOCAL_VAR(ip [1], argtype1) = LOCAL_VAR(ip [2], argtype2); \
    ip += 3;
                // When loading from a local, we might need to sign / zero extend to 4 bytes
                // which is our minimum "register" size in interp. They are only needed when
                // the address of the local is taken and we should try to optimize them out
                // because the local can't be propagated.
                case INTOP_MOV_I4_I1: MOV(int32_t, int8_t); break;
                case INTOP_MOV_I4_U1: MOV(int32_t, uint8_t); break;
                case INTOP_MOV_I4_I2: MOV(int32_t, int16_t); break;
                case INTOP_MOV_I4_U2: MOV(int32_t, uint16_t); break;
                // Normal moves between vars
                case INTOP_MOV_4: MOV(int32_t, int32_t); break;
                case INTOP_MOV_8: MOV(int64_t, int64_t); break;

                case INTOP_MOV_VT:
                    memmove(stack + ip[1], stack + ip[2], ip[3]);
                    ip += 4;
                    break;

                case INTOP_CONV_R_UN_I4:
                    LOCAL_VAR(ip[1], double) = (double)LOCAL_VAR(ip[2], uint32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_R_UN_I8:
                    LOCAL_VAR(ip[1], double) = (double)LOCAL_VAR(ip[2], uint64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I1_I4:
                    LOCAL_VAR(ip[1], int32_t) = (int8_t)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I1_I8:
                    LOCAL_VAR(ip[1], int32_t) = (int8_t)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I1_R4:
                    ConvFpHelper<int8_t, int32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I1_R8:
                    ConvFpHelper<int8_t, int32_t, double>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U1_I4:
                    LOCAL_VAR(ip[1], int32_t) = (uint8_t)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_U1_I8:
                    LOCAL_VAR(ip[1], int32_t) = (uint8_t)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_U1_R4:
                    ConvFpHelper<uint8_t, uint32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U1_R8:
                    ConvFpHelper<uint8_t, uint32_t, double>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I2_I4:
                    LOCAL_VAR(ip[1], int32_t) = (int16_t)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I2_I8:
                    LOCAL_VAR(ip[1], int32_t) = (int16_t)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I2_R4:
                    ConvFpHelper<int16_t, int32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I2_R8:
                    ConvFpHelper<int16_t, int32_t, double>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U2_I4:
                    LOCAL_VAR(ip[1], int32_t) = (uint16_t)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_U2_I8:
                    LOCAL_VAR(ip[1], int32_t) = (uint16_t)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_U2_R4:
                    ConvFpHelper<uint16_t, uint32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U2_R8:
                    ConvFpHelper<uint16_t, uint32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I4_R4:
                    ConvFpHelper<int32_t, int32_t, float>(stack, ip);
                    ip += 3;
                    break;;
                case INTOP_CONV_I4_R8:
                    ConvFpHelper<int32_t, int32_t, double>(stack, ip);
                    ip += 3;
                    break;;
                case INTOP_CONV_U4_R4:
                    ConvFpHelper<uint32_t, uint32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U4_R8:
                    ConvFpHelper<uint32_t, uint32_t, double>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I8_I4:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_I8_U4:
                    LOCAL_VAR(ip[1], int64_t) = (uint32_t)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;;
                case INTOP_CONV_I8_R4:
                    ConvFpHelper<int64_t, int64_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_I8_R8:
                    ConvFpHelper<int64_t, int64_t, double>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_R4_I4:
                    LOCAL_VAR(ip[1], float) = (float)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_R4_I8:
                    LOCAL_VAR(ip[1], float) = (float)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_R4_R8:
                    LOCAL_VAR(ip[1], float) = (float)LOCAL_VAR(ip[2], double);
                    ip += 3;
                    break;
                case INTOP_CONV_R8_I4:
                    LOCAL_VAR(ip[1], double) = (double)LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_R8_I8:
                    LOCAL_VAR(ip[1], double) = (double)LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_CONV_R8_R4:
                    LOCAL_VAR(ip[1], double) = (double)LOCAL_VAR(ip[2], float);
                    ip += 3;
                    break;
                case INTOP_CONV_U8_U4:
                    LOCAL_VAR(ip[1], uint64_t) = LOCAL_VAR(ip[2], uint32_t);
                    ip += 3;
                    break;
                case INTOP_CONV_U8_R4:
                    ConvFpHelper<uint64_t, uint64_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_U8_R8:
                    ConvFpHelper<uint64_t, uint64_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I1_I4:
                    ConvOvfHelper<int8_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I1_I8:
                    ConvOvfHelper<int8_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I1_R4:
                    ConvOvfFpHelper<int8_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I1_R8:
                    ConvOvfFpHelper<int8_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U1_I4:
                    ConvOvfHelper<uint8_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U1_I8:
                    ConvOvfHelper<uint8_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U1_R4:
                    ConvOvfFpHelper<uint8_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U1_R8:
                    ConvOvfFpHelper<uint8_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I2_I4:
                    ConvOvfHelper<int16_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I2_I8:
                    ConvOvfHelper<int16_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I2_R4:
                    ConvOvfFpHelper<int16_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I2_R8:
                    ConvOvfFpHelper<int16_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U2_I4:
                    ConvOvfHelper<uint16_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U2_I8:
                    ConvOvfHelper<uint16_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U2_R4:
                    ConvOvfFpHelper<uint16_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U2_R8:
                    ConvOvfFpHelper<uint16_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I4_U4:
                    ConvOvfHelper<int32_t, uint32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I4_I8:
                    ConvOvfHelper<int32_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I4_R4:
                    ConvOvfFpHelper<int32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I4_R8:
                    ConvOvfFpHelper<int32_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U4_I4:
                    ConvOvfHelper<uint32_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U4_I8:
                    ConvOvfHelper<uint32_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U4_R4:
                    ConvOvfFpHelper<uint32_t, float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U4_R8:
                    ConvOvfFpHelper<uint32_t, double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I8_U8:
                    ConvOvfHelper<int64_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I8_R4:
                    ConvOvfFpHelperI64<float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I8_R8:
                    ConvOvfFpHelperI64<double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U8_I4:
                    ConvOvfHelper<uint64_t, int32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U8_I8:
                    ConvOvfHelper<uint64_t, int64_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U8_R4:
                    ConvOvfFpHelperU64<float>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U8_R8:
                    ConvOvfFpHelperU64<double>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I1_U4:
                    ConvOvfHelper<int8_t, uint32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I1_U8:
                    ConvOvfHelper<int8_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U1_U4:
                    ConvOvfHelper<uint8_t, uint32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U1_U8:
                    ConvOvfHelper<uint8_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I2_U4:
                    ConvOvfHelper<int16_t, uint32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_I2_U8:
                    ConvOvfHelper<int16_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U2_U4:
                    ConvOvfHelper<uint16_t, uint32_t>(stack, ip);
                    ip += 3;
                    break;
                case INTOP_CONV_OVF_U2_U8:
                    ConvOvfHelper<uint16_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_I4_U8:
                    ConvOvfHelper<int32_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_CONV_OVF_U4_U8:
                    ConvOvfHelper<uint32_t, uint64_t>(stack, ip);
                    ip += 3;
                    break;

                case INTOP_SWITCH:
                {
                    uint32_t val = LOCAL_VAR(ip[1], uint32_t);
                    uint32_t n = ip[2];
                    ip += 3;
                    if (val < n)
                    {
                        ip += val;
                        ip += *ip;
                    }
                    else
                    {
                        ip += n;
                    }
                    break;
                }

                case INTOP_SAFEPOINT:
                    if (g_TrapReturningThreads)
                        JIT_PollGC();
                    ip++;
                    break;

                case INTOP_BR:
                    ip += ip[1];
                    break;

#define BR_UNOP(datatype, op)           \
    if (LOCAL_VAR(ip[1], datatype) op)  \
        ip += ip[2];                    \
    else \
        ip += 3;

                case INTOP_BRFALSE_I4:
                    BR_UNOP(int32_t, == 0);
                    break;
                case INTOP_BRFALSE_I8:
                    BR_UNOP(int64_t, == 0);
                    break;
                case INTOP_BRTRUE_I4:
                    BR_UNOP(int32_t, != 0);
                    break;
                case INTOP_BRTRUE_I8:
                    BR_UNOP(int64_t, != 0);
                    break;

#define BR_BINOP_COND(cond) \
    if (cond)               \
        ip += ip[3];        \
    else                    \
        ip += 4;

#define BR_BINOP(datatype, op) \
    BR_BINOP_COND(LOCAL_VAR(ip[1], datatype) op LOCAL_VAR(ip[2], datatype))

                case INTOP_BEQ_I4:
                    BR_BINOP(int32_t, ==);
                    break;
                case INTOP_BEQ_I8:
                    BR_BINOP(int64_t, ==);
                    break;
                case INTOP_BEQ_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(!isunordered(f1, f2) && f1 == f2);
                    break;
                }
                case INTOP_BEQ_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(!isunordered(d1, d2) && d1 == d2);
                    break;
                }
                case INTOP_BGE_I4:
                    BR_BINOP(int32_t, >=);
                    break;
                case INTOP_BGE_I8:
                    BR_BINOP(int64_t, >=);
                    break;
                case INTOP_BGE_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(!isunordered(f1, f2) && f1 >= f2);
                    break;
                }
                case INTOP_BGE_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(!isunordered(d1, d2) && d1 >= d2);
                    break;
                }
                case INTOP_BGT_I4:
                    BR_BINOP(int32_t, >);
                    break;
                case INTOP_BGT_I8:
                    BR_BINOP(int64_t, >);
                    break;
                case INTOP_BGT_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(!isunordered(f1, f2) && f1 > f2);
                    break;
                }
                case INTOP_BGT_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(!isunordered(d1, d2) && d1 > d2);
                    break;
                }
                case INTOP_BLT_I4:
                    BR_BINOP(int32_t, <);
                    break;
                case INTOP_BLT_I8:
                    BR_BINOP(int64_t, <);
                    break;
                case INTOP_BLT_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(!isunordered(f1, f2) && f1 < f2);
                    break;
                }
                case INTOP_BLT_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(!isunordered(d1, d2) && d1 < d2);
                    break;
                }
                case INTOP_BLE_I4:
                    BR_BINOP(int32_t, <=);
                    break;
                case INTOP_BLE_I8:
                    BR_BINOP(int64_t, <=);
                    break;
                case INTOP_BLE_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(!isunordered(f1, f2) && f1 <= f2);
                    break;
                }
                case INTOP_BLE_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(!isunordered(d1, d2) && d1 <= d2);
                    break;
                }
                case INTOP_BNE_UN_I4:
                    BR_BINOP(uint32_t, !=);
                    break;
                case INTOP_BNE_UN_I8:
                    BR_BINOP(uint64_t, !=);
                    break;
                case INTOP_BNE_UN_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(isunordered(f1, f2) || f1 != f2);
                    break;
                }
                case INTOP_BNE_UN_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(isunordered(d1, d2) || d1 != d2);
                    break;
                }
                case INTOP_BGE_UN_I4:
                    BR_BINOP(uint32_t, >=);
                    break;
                case INTOP_BGE_UN_I8:
                    BR_BINOP(uint64_t, >=);
                    break;
                case INTOP_BGE_UN_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(isunordered(f1, f2) || f1 >= f2);
                    break;
                }
                case INTOP_BGE_UN_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(isunordered(d1, d2) || d1 >= d2);
                    break;
                }
                case INTOP_BGT_UN_I4:
                    BR_BINOP(uint32_t, >);
                    break;
                case INTOP_BGT_UN_I8:
                    BR_BINOP(uint64_t, >);
                    break;
                case INTOP_BGT_UN_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(isunordered(f1, f2) || f1 > f2);
                    break;
                }
                case INTOP_BGT_UN_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(isunordered(d1, d2) || d1 > d2);
                    break;
                }
                case INTOP_BLE_UN_I4:
                    BR_BINOP(uint32_t, <=);
                    break;
                case INTOP_BLE_UN_I8:
                    BR_BINOP(uint64_t, <=);
                    break;
                case INTOP_BLE_UN_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(isunordered(f1, f2) || f1 <= f2);
                    break;
                }
                case INTOP_BLE_UN_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(isunordered(d1, d2) || d1 <= d2);
                    break;
                }
                case INTOP_BLT_UN_I4:
                    BR_BINOP(uint32_t, <);
                    break;
                case INTOP_BLT_UN_I8:
                    BR_BINOP(uint64_t, <);
                    break;
                case INTOP_BLT_UN_R4:
                {
                    float f1 = LOCAL_VAR(ip[1], float);
                    float f2 = LOCAL_VAR(ip[2], float);
                    BR_BINOP_COND(isunordered(f1, f2) || f1 < f2);
                    break;
                }
                case INTOP_BLT_UN_R8:
                {
                    double d1 = LOCAL_VAR(ip[1], double);
                    double d2 = LOCAL_VAR(ip[2], double);
                    BR_BINOP_COND(isunordered(d1, d2) || d1 < d2);
                    break;
                }

                case INTOP_ADD_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) + LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_ADD_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) + LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_ADD_R4:
                    LOCAL_VAR(ip[1], float) = LOCAL_VAR(ip[2], float) + LOCAL_VAR(ip[3], float);
                    ip += 4;
                    break;
                case INTOP_ADD_R8:
                    LOCAL_VAR(ip[1], double) = LOCAL_VAR(ip[2], double) + LOCAL_VAR(ip[3], double);
                    ip += 4;
                    break;
                case INTOP_ADD_I4_IMM:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) + ip[3];
                    ip += 4;
                    break;
                case INTOP_ADD_I8_IMM:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) + ip[3];
                    ip += 4;
                    break;
                case INTOP_ADD_OVF_I4:
                {
                    int32_t i1 = LOCAL_VAR(ip[2], int32_t);
                    int32_t i2 = LOCAL_VAR(ip[3], int32_t);
                    int32_t i3;
                    if (!ClrSafeInt<int32_t>::addition(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int32_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_ADD_OVF_I8:
                {
                    int64_t i1 = LOCAL_VAR(ip[2], int64_t);
                    int64_t i2 = LOCAL_VAR(ip[3], int64_t);
                    int64_t i3;
                    if (!ClrSafeInt<int64_t>::addition(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int64_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_ADD_OVF_UN_I4:
                {
                    uint32_t i1 = LOCAL_VAR(ip[2], uint32_t);
                    uint32_t i2 = LOCAL_VAR(ip[3], uint32_t);
                    uint32_t i3;
                    if (!ClrSafeInt<uint32_t>::addition(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint32_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_ADD_OVF_UN_I8:
                {
                    uint64_t i1 = LOCAL_VAR(ip[2], uint64_t);
                    uint64_t i2 = LOCAL_VAR(ip[3], uint64_t);
                    uint64_t i3;
                    if (!ClrSafeInt<uint64_t>::addition(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint64_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_SUB_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) - LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SUB_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) - LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_SUB_R4:
                    LOCAL_VAR(ip[1], float) = LOCAL_VAR(ip[2], float) - LOCAL_VAR(ip[3], float);
                    ip += 4;
                    break;
                case INTOP_SUB_R8:
                    LOCAL_VAR(ip[1], double) = LOCAL_VAR(ip[2], double) - LOCAL_VAR(ip[3], double);
                    ip += 4;
                    break;

                case INTOP_SUB_OVF_I4:
                {
                    int32_t i1 = LOCAL_VAR(ip[2], int32_t);
                    int32_t i2 = LOCAL_VAR(ip[3], int32_t);
                    int32_t i3;
                    if (!ClrSafeInt<int32_t>::subtraction(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int32_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_SUB_OVF_I8:
                {
                    int64_t i1 = LOCAL_VAR(ip[2], int64_t);
                    int64_t i2 = LOCAL_VAR(ip[3], int64_t);
                    int64_t i3;
                    if (!ClrSafeInt<int64_t>::subtraction(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int64_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_SUB_OVF_UN_I4:
                {
                    uint32_t i1 = LOCAL_VAR(ip[2], uint32_t);
                    uint32_t i2 = LOCAL_VAR(ip[3], uint32_t);
                    uint32_t i3;
                    if (!ClrSafeInt<uint32_t>::subtraction(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint32_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_SUB_OVF_UN_I8:
                {
                    uint64_t i1 = LOCAL_VAR(ip[2], uint64_t);
                    uint64_t i2 = LOCAL_VAR(ip[3], uint64_t);
                    uint64_t i3;
                    if (!ClrSafeInt<uint64_t>::subtraction(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint64_t) = i3;
                    ip += 4;
                    break;
                }

                case INTOP_MUL_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) * LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_MUL_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) * LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_MUL_R4:
                    LOCAL_VAR(ip[1], float) = LOCAL_VAR(ip[2], float) * LOCAL_VAR(ip[3], float);
                    ip += 4;
                    break;
                case INTOP_MUL_R8:
                    LOCAL_VAR(ip[1], double) = LOCAL_VAR(ip[2], double) * LOCAL_VAR(ip[3], double);
                    ip += 4;
                    break;
                case INTOP_MUL_OVF_I4:
                {
                    int32_t i1 = LOCAL_VAR(ip[2], int32_t);
                    int32_t i2 = LOCAL_VAR(ip[3], int32_t);
                    int32_t i3;
                    if (!ClrSafeInt<int32_t>::multiply(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int32_t) = i3;
                    ip += 4;
                    break;
                }

                case INTOP_MUL_OVF_I8:
                {
                    int64_t i1 = LOCAL_VAR(ip[2], int64_t);
                    int64_t i2 = LOCAL_VAR(ip[3], int64_t);
                    int64_t i3;
                    if (!ClrSafeInt<int64_t>::multiply(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int64_t) = i3;
                    ip += 4;
                    break;
                }

                case INTOP_MUL_OVF_UN_I4:
                {
                    uint32_t i1 = LOCAL_VAR(ip[2], uint32_t);
                    uint32_t i2 = LOCAL_VAR(ip[3], uint32_t);
                    uint32_t i3;
                    if (!ClrSafeInt<uint32_t>::multiply(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint32_t) = i3;
                    ip += 4;
                    break;
                }

                case INTOP_MUL_OVF_UN_I8:
                {
                    uint64_t i1 = LOCAL_VAR(ip[2], uint64_t);
                    uint64_t i2 = LOCAL_VAR(ip[3], uint64_t);
                    uint64_t i3;
                    if (!ClrSafeInt<uint64_t>::multiply(i1, i2, i3))
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], uint64_t) = i3;
                    ip += 4;
                    break;
                }
                case INTOP_DIV_I4:
                {
                    int32_t i1 = LOCAL_VAR(ip[2], int32_t);
                    int32_t i2 = LOCAL_VAR(ip[3], int32_t);
                    if (i2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    if (i2 == -1 && i1 == INT32_MIN)
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int32_t) = i1 / i2;
                    ip += 4;
                    break;
                }
                case INTOP_DIV_I8:
                {
                    int64_t l1 = LOCAL_VAR(ip[2], int64_t);
                    int64_t l2 = LOCAL_VAR(ip[3], int64_t);
                    if (l2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    if (l2 == -1 && l1 == INT64_MIN)
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int64_t) = l1 / l2;
                    ip += 4;
                    break;
                }
                case INTOP_DIV_R4:
                    LOCAL_VAR(ip[1], float) = LOCAL_VAR(ip[2], float) / LOCAL_VAR(ip[3], float);
                    ip += 4;
                    break;
                case INTOP_DIV_R8:
                    LOCAL_VAR(ip[1], double) = LOCAL_VAR(ip[2], double) / LOCAL_VAR(ip[3], double);
                    ip += 4;
                    break;
                case INTOP_DIV_UN_I4:
                {
                    uint32_t i2 = LOCAL_VAR(ip[3], uint32_t);
                    if (i2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    LOCAL_VAR(ip[1], uint32_t) = LOCAL_VAR(ip[2], uint32_t) / i2;
                    ip += 4;
                    break;
                }
                case INTOP_DIV_UN_I8:
                {
                    uint64_t l2 = LOCAL_VAR(ip[3], uint64_t);
                    if (l2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    LOCAL_VAR(ip[1], uint64_t) = LOCAL_VAR(ip[2], uint64_t) / l2;
                    ip += 4;
                    break;
                }

                case INTOP_REM_I4:
                {
                    int32_t i1 = LOCAL_VAR(ip[2], int32_t);
                    int32_t i2 = LOCAL_VAR(ip[3], int32_t);
                    if (i2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    if (i2 == -1 && i1 == INT32_MIN)
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int32_t) = i1 % i2;
                    ip += 4;
                    break;
                }
                case INTOP_REM_I8:
                {
                    int64_t l1 = LOCAL_VAR(ip[2], int64_t);
                    int64_t l2 = LOCAL_VAR(ip[3], int64_t);
                    if (l2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    if (l2 == -1 && l1 == INT64_MIN)
                        COMPlusThrow(kOverflowException);
                    LOCAL_VAR(ip[1], int64_t) = l1 % l2;
                    ip += 4;
                    break;
                }
                case INTOP_REM_R4:
                    LOCAL_VAR(ip[1], float) = fmodf(LOCAL_VAR(ip[2], float), LOCAL_VAR(ip[3], float));
                    ip += 4;
                    break;
                case INTOP_REM_R8:
                    LOCAL_VAR(ip[1], double) = fmod(LOCAL_VAR(ip[2], double), LOCAL_VAR(ip[3], double));
                    ip += 4;
                    break;
                case INTOP_REM_UN_I4:
                {
                    uint32_t i2 = LOCAL_VAR(ip[3], uint32_t);
                    if (i2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    LOCAL_VAR(ip[1], uint32_t) = LOCAL_VAR(ip[2], uint32_t) % i2;
                    ip += 4;
                    break;
                }
                case INTOP_REM_UN_I8:
                {
                    uint64_t l2 = LOCAL_VAR(ip[3], uint64_t);
                    if (l2 == 0)
                        COMPlusThrow(kDivideByZeroException);
                    LOCAL_VAR(ip[1], uint64_t) = LOCAL_VAR(ip[2], uint64_t) % l2;
                    ip += 4;
                    break;
                }

                case INTOP_SHL_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) << LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SHL_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) << LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SHR_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) >> LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SHR_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) >> LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SHR_UN_I4:
                    LOCAL_VAR(ip[1], uint32_t) = LOCAL_VAR(ip[2], uint32_t) >> LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_SHR_UN_I8:
                    LOCAL_VAR(ip[1], uint64_t) = LOCAL_VAR(ip[2], uint64_t) >> LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;

                case INTOP_NEG_I4:
                    LOCAL_VAR(ip[1], int32_t) = - LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_NEG_I8:
                    LOCAL_VAR(ip[1], int64_t) = - LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;
                case INTOP_NEG_R4:
                    LOCAL_VAR(ip[1], float) = - LOCAL_VAR(ip[2], float);
                    ip += 3;
                    break;
                case INTOP_NEG_R8:
                    LOCAL_VAR(ip[1], double) = - LOCAL_VAR(ip[2], double);
                    ip += 3;
                    break;
                case INTOP_NOT_I4:
                    LOCAL_VAR(ip[1], int32_t) = ~ LOCAL_VAR(ip[2], int32_t);
                    ip += 3;
                    break;
                case INTOP_NOT_I8:
                    LOCAL_VAR(ip[1], int64_t) = ~ LOCAL_VAR(ip[2], int64_t);
                    ip += 3;
                    break;

                case INTOP_AND_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) & LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_AND_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) & LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_OR_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) | LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_OR_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) | LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_XOR_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) ^ LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_XOR_I8:
                    LOCAL_VAR(ip[1], int64_t) = LOCAL_VAR(ip[2], int64_t) ^ LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;

#define CMP_BINOP_FP(datatype, op, noOrderVal)      \
    do {                                            \
        datatype f1 = LOCAL_VAR(ip[2], datatype);   \
        datatype f2 = LOCAL_VAR(ip[3], datatype);   \
        if (isunordered(f1, f2))                    \
            LOCAL_VAR(ip[1], int32_t) = noOrderVal; \
        else                                        \
            LOCAL_VAR(ip[1], int32_t) = f1 op f2;   \
        ip += 4;                                    \
    } while (0)

                case INTOP_CEQ_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) == LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_CEQ_I8:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int64_t) == LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_CEQ_R4:
                    CMP_BINOP_FP(float, ==, 0);
                    break;
                case INTOP_CEQ_R8:
                    CMP_BINOP_FP(double, ==, 0);
                    break;

                case INTOP_CGT_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) > LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_CGT_I8:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int64_t) > LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_CGT_R4:
                    CMP_BINOP_FP(float, >, 0);
                    break;
                case INTOP_CGT_R8:
                    CMP_BINOP_FP(double, >, 0);
                    break;

                case INTOP_CGT_UN_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], uint32_t) > LOCAL_VAR(ip[3], uint32_t);
                    ip += 4;
                    break;
                case INTOP_CGT_UN_I8:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], uint64_t) > LOCAL_VAR(ip[3], uint64_t);
                    ip += 4;
                    break;
                case INTOP_CGT_UN_R4:
                    CMP_BINOP_FP(float, >, 1);
                    break;
                case INTOP_CGT_UN_R8:
                    CMP_BINOP_FP(double, >, 1);
                    break;

                case INTOP_CLT_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int32_t) < LOCAL_VAR(ip[3], int32_t);
                    ip += 4;
                    break;
                case INTOP_CLT_I8:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], int64_t) < LOCAL_VAR(ip[3], int64_t);
                    ip += 4;
                    break;
                case INTOP_CLT_R4:
                    CMP_BINOP_FP(float, <, 0);
                    break;
                case INTOP_CLT_R8:
                    CMP_BINOP_FP(double, <, 0);
                    break;

                case INTOP_CLT_UN_I4:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], uint32_t) < LOCAL_VAR(ip[3], uint32_t);
                    ip += 4;
                    break;
                case INTOP_CLT_UN_I8:
                    LOCAL_VAR(ip[1], int32_t) = LOCAL_VAR(ip[2], uint64_t) < LOCAL_VAR(ip[3], uint64_t);
                    ip += 4;
                    break;
                case INTOP_CLT_UN_R4:
                    CMP_BINOP_FP(float, <, 1);
                    break;
                case INTOP_CLT_UN_R8:
                    CMP_BINOP_FP(double, <, 1);
                    break;

#define LDIND(dtype, ftype)                                 \
    do {                                                    \
        char *src = LOCAL_VAR(ip[2], char*);                \
        NULL_CHECK(src);                                    \
        LOCAL_VAR(ip[1], dtype) = *(ftype*)(src + ip[3]);   \
        ip += 4;                                            \
    } while (0)

                case INTOP_LDIND_I1:
                    LDIND(int32_t, int8_t);
                    break;
                case INTOP_LDIND_U1:
                    LDIND(int32_t, uint8_t);
                    break;
                case INTOP_LDIND_I2:
                    LDIND(int32_t, int16_t);
                    break;
                case INTOP_LDIND_U2:
                    LDIND(int32_t, uint16_t);
                    break;
                case INTOP_LDIND_I4:
                    LDIND(int32_t, int32_t);
                    break;
                case INTOP_LDIND_I8:
                    LDIND(int64_t, int64_t);
                    break;
                case INTOP_LDIND_R4:
                    LDIND(float, float);
                    break;
                case INTOP_LDIND_R8:
                    LDIND(double, double);
                    break;
                case INTOP_LDIND_VT:
                {
                    char *src = LOCAL_VAR(ip[2], char*);
                    NULL_CHECK(src);
                    memcpy(stack + ip[1], (char*)src + ip[3], ip[4]);
                    ip += 5;
                    break;
                }

#define STIND(dtype, ftype)                                         \
    do                                                              \
    {                                                               \
        char *dst = LOCAL_VAR(ip[1], char*);                        \
        NULL_CHECK(dst);                                            \
        *(ftype*)(dst + ip[3]) = (ftype)(LOCAL_VAR(ip[2], dtype));  \
        ip += 4;                                                    \
    } while (0)

                case INTOP_STIND_I1:
                    STIND(int32_t, int8_t);
                    break;
                case INTOP_STIND_U1:
                    STIND(int32_t, uint8_t);
                    break;
                case INTOP_STIND_I2:
                    STIND(int32_t, int16_t);
                    break;
                case INTOP_STIND_U2:
                    STIND(int32_t, uint16_t);
                    break;
                case INTOP_STIND_I4:
                    STIND(int32_t, int32_t);
                    break;
                case INTOP_STIND_I8:
                    STIND(int64_t, int64_t);
                    break;
                case INTOP_STIND_R4:
                    STIND(float, float);
                    break;
                case INTOP_STIND_R8:
                    STIND(double, double);
                    break;
                case INTOP_STIND_O:
                {
                    char *dst = LOCAL_VAR(ip[1], char*);
                    OBJECTREF storeObj = LOCAL_VAR(ip[2], OBJECTREF);
                    NULL_CHECK(dst);
                    SetObjectReferenceUnchecked((OBJECTREF*)(dst + ip[3]), storeObj);
                    ip += 4;
                    break;
                }
                case INTOP_STIND_VT_NOREF:
                {
                    char *dest = LOCAL_VAR(ip[1], char*);
                    NULL_CHECK(dest);
                    memcpyNoGCRefs(dest + ip[3], stack + ip[2], ip[4]);
                    ip += 5;
                    break;
                }
                case INTOP_STIND_VT:
                {
                    MethodTable *pMT = (MethodTable*)pMethod->pDataItems[ip[4]];
                    char *dest = LOCAL_VAR(ip[1], char*);
                    NULL_CHECK(dest);
                    CopyValueClassUnchecked(dest + ip[3], stack + ip[2], pMT);
                    ip += 5;
                    break;
                }
                case INTOP_LDFLDA:
                {
                    char *src = LOCAL_VAR(ip[2], char*);
                    NULL_CHECK(src);
                    LOCAL_VAR(ip[1], char*) = src + ip[3];
                    ip += 4;
                    break;
                }

                case INTOP_CALL_HELPER_P_P:
                {
                    HELPER_FTN_P_P helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_P>(pMethod, ip[2]);
                    void* helperArg = pMethod->pDataItems[ip[3]];

                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg);
                    ip += 4;
                    break;
                }

                case INTOP_CALL_HELPER_P_S:
                {
                    HELPER_FTN_P_P helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_P>(pMethod, ip[3]);
                    void* helperArg = LOCAL_VAR(ip[2], void*);

                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg);
                    ip += 4;
                    break;
                }

                case INTOP_CALL_HELPER_P_PS:
                {
                    HELPER_FTN_P_PP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_PP>(pMethod, ip[3]);
                    void* helperArg = pMethod->pDataItems[ip[4]];

                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg, LOCAL_VAR(ip[2], void*));
                    ip += 5;
                    break;
                }

                case INTOP_CALL_HELPER_P_SP:
                {
                    HELPER_FTN_P_PP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_PP>(pMethod, ip[3]);
                    void* helperArg = pMethod->pDataItems[ip[4]];

                    LOCAL_VAR(ip[1], void*) = helperFtn(LOCAL_VAR(ip[2], void*), helperArg);
                    ip += 5;
                    break;
                }

                case INTOP_CALL_HELPER_P_G:
                {
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[4]];
                    void* helperArg = DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_P_P helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_P>(pMethod, ip[3]);

                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg);
                    ip += 5;
                    break;
                }

                case INTOP_CALL_HELPER_P_GS:
                {
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    void* helperArg = DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_P_PP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_PP>(pMethod, ip[4]);

                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg, LOCAL_VAR(ip[3], void*));
                    ip += 6;
                    break;
                }

                case INTOP_CALL_HELPER_P_GA:
                {
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    void* helperArg = DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_P_PP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_PP>(pMethod, ip[4]);
                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg, LOCAL_VAR_ADDR(ip[3], void*));
                    ip += 6;
                    break;
                }

                case INTOP_CALL_HELPER_P_PA:
                {
                    HELPER_FTN_P_PP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_P_PP>(pMethod, ip[3]);
                    void* helperArg = pMethod->pDataItems[ip[4]];
                    LOCAL_VAR(ip[1], void*) = helperFtn(helperArg, LOCAL_VAR_ADDR(ip[2], void*));
                    ip += 5;
                    break;
                }

                case INTOP_CALL_HELPER_V_AGS:
                {
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    void* helperArg = DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_V_PPP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_V_PPP>(pMethod, ip[4]);
                    helperFtn(LOCAL_VAR_ADDR(ip[1], void*), helperArg, LOCAL_VAR(ip[3], void*));
                    ip += 6;
                    break;
                }

                case INTOP_CALL_HELPER_V_APS:
                {
                    HELPER_FTN_V_PPP helperFtn = GetPossiblyIndirectHelper<HELPER_FTN_V_PPP>(pMethod, ip[3]);
                    void* helperArg = pMethod->pDataItems[ip[4]];
                    helperFtn(LOCAL_VAR_ADDR(ip[1], void*), helperArg, LOCAL_VAR(ip[2], void*));
                    ip += 5;
                    break;
                }

                case INTOP_CALLVIRT:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];

                    MethodDesc *pMD = (MethodDesc*)pMethod->pDataItems[methodSlot];

                    OBJECTREF *pThisArg = LOCAL_VAR_ADDR(callArgsOffset, OBJECTREF);
                    NULL_CHECK(*pThisArg);

                    // Interpreter-TODO
                    // This needs to be optimized, not operating at MethodDesc level, rather with ftnptr
                    // slots containing the interpreter IR pointer
                    targetMethod = pMD->GetMethodDescOfVirtualizedCode(pThisArg, pMD->GetMethodTable());
                    ip += 4;
                    goto CALL_INTERP_METHOD;
                }

                case INTOP_CALLI:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    int32_t calliFunctionPointerVar = ip[3];
                    int32_t calliCookie = ip[4];

                    CallStubHeader *pCallStub = (CallStubHeader*)pMethod->pDataItems[calliCookie];
                    ip += 5;

                    InvokeCalliStub(LOCAL_VAR(calliFunctionPointerVar, PCODE), pCallStub, stack + callArgsOffset, stack + returnOffset);
                    break;
                }

                case INTOP_CALL_PINVOKE:
                {
                    // This opcode handles p/invokes that don't use a managed wrapper for marshaling. These
                    //  calls are special in that they need an InlinedCallFrame in order for proper EH to happen

                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];
                    int32_t targetAddrSlot = ip[4];
                    int32_t flags = ip[5];

                    ip += 6;
                    targetMethod = (MethodDesc*)pMethod->pDataItems[methodSlot];
                    PCODE callTarget = (flags & (int32_t)PInvokeCallFlags::Indirect)
                        ? *(PCODE *)pMethod->pDataItems[targetAddrSlot]
                        : (PCODE)pMethod->pDataItems[targetAddrSlot];

                    InlinedCallFrame inlinedCallFrame;
                    inlinedCallFrame.m_pCallerReturnAddress = (TADDR)ip;
                    inlinedCallFrame.m_pCallSiteSP = pFrame;
                    inlinedCallFrame.m_pCalleeSavedFP = (TADDR)stack;
                    inlinedCallFrame.m_pThread = GetThread();
                    inlinedCallFrame.m_Datum = NULL;
                    inlinedCallFrame.Push();

                    {
                        GCX_MAYBE_PREEMP(!(flags & (int32_t)PInvokeCallFlags::SuppressGCTransition));
                        InvokeCompiledMethod(targetMethod, stack + callArgsOffset, stack + returnOffset, callTarget);
                    }

                    inlinedCallFrame.Pop();

                    break;
                }

                case INTOP_CALLDELEGATE:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];

                    // targetMethod holds a pointer to the Invoke method of the delegate, not the final actual target.
                    targetMethod = (MethodDesc*)pMethod->pDataItems[methodSlot];

                    ip += 4;

                    DELEGATEREF delegateObj = LOCAL_VAR(callArgsOffset, DELEGATEREF);
                    NULL_CHECK(delegateObj);
                    PCODE targetAddress = delegateObj->GetMethodPtr();
                    OBJECTREF targetMethodObj = delegateObj->GetTarget();
                    LOCAL_VAR(callArgsOffset, OBJECTREF) = targetMethodObj;

                    // TODO! Once we are investigating performance here, we may want to optimize this so that
                    // delegate calls to interpeted methods don't have to go through the native invoke here, but for
                    // now this should work well.
                    InvokeDelegateInvokeMethod(targetMethod, stack + callArgsOffset, stack + returnOffset, targetAddress);
                    break;
                }

                case INTOP_CALL:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];

                    ip += 4;
CALL_INTERP_SLOT:
                    targetMethod = (MethodDesc*)pMethod->pDataItems[methodSlot];
CALL_INTERP_METHOD:
                    InterpByteCodeStart* targetIp = targetMethod->GetInterpreterCode();
                    if (targetIp == NULL)
                    {
                        {
                            // This is an optimization to ensure that the stack walk will not have to search
                            // for the topmost frame in the current InterpExecMethod. It is not required
                            // for correctness, as the stack walk will find the topmost frame anyway. But it
                            // would need to seek through the frames to find it.
                            // An alternative approach would be to update the topmost frame during stack walk
                            // to make the probability that the next stack walk will need to search only a
                            // small subset of frames high.
                            pInterpreterFrame->SetTopInterpMethodContextFrame(pFrame);
                            GCX_PREEMP();
                            // Attempt to setup the interpreter code for the target method.
                            if ((targetMethod->IsIL() || targetMethod->IsNoMetadata()) && !targetMethod->IsUnboxingStub())
                            {
                                targetMethod->PrepareInitialCode(CallerGCMode::Coop);
                            }
                            targetIp = targetMethod->GetInterpreterCode();
                        }
                        if (targetIp == NULL)
                        {
                            // If we didn't get the interpreter code pointer setup, then this is a method we need to invoke as a compiled method.
                            InvokeCompiledMethod(targetMethod, stack + callArgsOffset, stack + returnOffset, targetMethod->GetMultiCallableAddrOfCode(CORINFO_ACCESS_ANY));
                            break;
                        }
                    }

                    // Save current execution state for when we return from called method
                    pFrame->ip = ip;

                    // Allocate child frame.
                    {
                        InterpMethodContextFrame *pChildFrame = pFrame->pNext;
                        if (!pChildFrame)
                        {
                            pChildFrame = (InterpMethodContextFrame*)alloca(sizeof(InterpMethodContextFrame));
                            pChildFrame->pNext = NULL;
                            pFrame->pNext = pChildFrame;
                        }
                        pChildFrame->ReInit(pFrame, targetIp, stack + returnOffset, stack + callArgsOffset);
                        pFrame = pChildFrame;
                    }
                    assert (((size_t)pFrame->pStack % INTERP_STACK_ALIGNMENT) == 0);

                    // Set execution state for the new frame
                    pMethod = pFrame->startIp->Method;
                    assert(pMethod->CheckIntegrity());
                    stack = pFrame->pStack;
                    ip = pFrame->startIp->GetByteCodes();
                    pThreadContext->pStackPointer = stack + pMethod->allocaSize;
                    break;
                }
                case INTOP_NEWOBJ_GENERIC:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[4];
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    MethodTable *pMTNewObj = (MethodTable*)DoGenericLookup(LOCAL_VAR(ip[3], void*), pLookup);

                    OBJECTREF objRef = AllocateObject(pMTNewObj);

                    // This is return value
                    LOCAL_VAR(returnOffset, OBJECTREF) = objRef;
                    // Set `this` arg for ctor call
                    LOCAL_VAR (callArgsOffset, OBJECTREF) = objRef;
                    ip += 6;

                    goto CALL_INTERP_SLOT;
                }
                case INTOP_NEWOBJ:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];

                    OBJECTREF objRef = AllocateObject((MethodTable*)pMethod->pDataItems[ip[4]]);

                    // This is return value
                    LOCAL_VAR(returnOffset, OBJECTREF) = objRef;
                    // Set `this` arg for ctor call
                    LOCAL_VAR (callArgsOffset, OBJECTREF) = objRef;
                    ip += 5;

                    goto CALL_INTERP_SLOT;
                }
                case INTOP_NEWMDARR:
                {
                    LOCAL_VAR(ip[1], OBJECTREF) = CreateMultiDimArray((MethodTable*)pMethod->pDataItems[ip[3]], stack, ip[2], ip[4]);
                    ip += 5;
                    break;
                }
                case INTOP_NEWMDARR_GENERIC:
                {
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[4]];
                    MethodTable *pMTArray = (MethodTable*)DoGenericLookup(LOCAL_VAR(ip[3], void*), pLookup);

                    LOCAL_VAR(ip[1], OBJECTREF) = CreateMultiDimArray(pMTArray, stack, ip[2], ip[5]);
                    ip += 6;
                    break;
                }
                case INTOP_NEWOBJ_VT:
                {
                    returnOffset = ip[1];
                    callArgsOffset = ip[2];
                    methodSlot = ip[3];

                    int32_t vtSize = ip[4];
                    void *vtThis = stack + returnOffset;

                    // pass the address of the valuetype
                    LOCAL_VAR(callArgsOffset, void*) = vtThis;

                    ip += 5;
                    goto CALL_INTERP_SLOT;
                }
                case INTOP_ZEROBLK_IMM:
                    memset(LOCAL_VAR(ip[1], void*), 0, ip[2]);
                    ip += 3;
                    break;
                case INTOP_CPBLK:
                {
                    void* dst = LOCAL_VAR(ip[1], void*);
                    void* src = LOCAL_VAR(ip[2], void*);
                    uint32_t size = LOCAL_VAR(ip[3], uint32_t);
                    if (size && (!dst || !src))
                        COMPlusThrow(kNullReferenceException);
                    else
                        memcpyNoGCRefs(dst, src, size);
                    ip += 4;
                    break;
                }
                case INTOP_LOCALLOC:
                {
                    size_t len = LOCAL_VAR(ip[2], size_t);
                    void* pMemory = NULL;

                    if (len > 0)
                    {
                        pMemory = pThreadContext->frameDataAllocator.Alloc(pFrame, len);
                        if (pMemory == NULL)
                        {
                            COMPlusThrowOM();
                        }
                        if (pMethod->initLocals)
                        {
                            memset(pMemory, 0, len);
                        }
                    }

                    LOCAL_VAR(ip[1], void*) = pMemory;
                    ip += 3;
                    break;
                }
                case INTOP_THROW:
                {
                    OBJECTREF throwable;
                    if (LOCAL_VAR(ip[1], OBJECTREF) == nullptr)
                    {
                        EEException ex(kNullReferenceException);
                        throwable = ex.CreateThrowable();
                    }
                    else
                    {
                        throwable = LOCAL_VAR(ip[1], OBJECTREF);
                    }
                    pInterpreterFrame->SetIsFaulting(true);
                    DispatchManagedException(throwable);
                    UNREACHABLE();
                    break;
                }
                case INTOP_RETHROW:
                {
                    pInterpreterFrame->SetIsFaulting(true);
                    DispatchRethrownManagedException();
                    UNREACHABLE();
                    break;
                }
                case INTOP_LOAD_EXCEPTION:
                    // This opcode loads the exception object coming from a catch / filter funclet caller to a variable.
                    assert(pExceptionClauseArgs != NULL);
                    LOCAL_VAR(ip[1], OBJECTREF) = pExceptionClauseArgs->throwable;
                    ip += 2;
                    break;
                case INTOP_UNBOX_ANY:
                {
                    int opcode = *ip;
                    int dreg = ip[1];
                    int sreg = ip[2];
                    HELPER_FTN_BOX_UNBOX helper = GetPossiblyIndirectHelper<HELPER_FTN_BOX_UNBOX>(pMethod, ip[3]);
                    MethodTable *pMT = (MethodTable*)pMethod->pDataItems[ip[4]];

                    // private static ref byte Unbox(MethodTable* toTypeHnd, object obj)
                    Object *src = LOCAL_VAR(sreg, Object*);
                    void *unboxedData = helper(pMT, src);
                    CopyValueClassUnchecked(LOCAL_VAR_ADDR(dreg, void), unboxedData, pMT);

                    ip += 5;
                    break;
                }

                case INTOP_UNBOX_ANY_GENERIC:
                {
                    int opcode = *ip;
                    int dreg = ip[1];
                    int sreg = ip[3];
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    MethodTable *pMTBoxedObj = (MethodTable*)DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_BOX_UNBOX helper = GetPossiblyIndirectHelper<HELPER_FTN_BOX_UNBOX>(pMethod, ip[4]);

                    // private static ref byte Unbox(MethodTable* toTypeHnd, object obj)
                    Object *src = LOCAL_VAR(sreg, Object*);
                    void *unboxedData = helper(pMTBoxedObj, src);
                    CopyValueClassUnchecked(LOCAL_VAR_ADDR(dreg, void), unboxedData, pMTBoxedObj->IsNullable() ? pMTBoxedObj->GetInstantiation()[0].AsMethodTable() : pMTBoxedObj);

                    ip += 6;
                    break;
                }
                case INTOP_NEWARR:
                {
                    int32_t length = LOCAL_VAR(ip[2], int32_t);
                    if (length < 0)
                        COMPlusThrow(kArgumentOutOfRangeException);

                    MethodTable* arrayClsHnd = (MethodTable*)pMethod->pDataItems[ip[3]];
                    HELPER_FTN_NEWARR helper = GetPossiblyIndirectHelper<HELPER_FTN_NEWARR>(pMethod, ip[4]);

                    Object* arr = helper(arrayClsHnd, (intptr_t)length);
                    LOCAL_VAR(ip[1], OBJECTREF) = ObjectToOBJECTREF(arr);

                    ip += 5;
                    break;
                }
                case INTOP_NEWARR_GENERIC:
                {
                    int32_t length = LOCAL_VAR(ip[3], int32_t);
                    if (length < 0)
                        COMPlusThrow(kArgumentOutOfRangeException);

                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    MethodTable *arrayClsHnd = (MethodTable*)DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);

                    HELPER_FTN_NEWARR helper = GetPossiblyIndirectHelper<HELPER_FTN_NEWARR>(pMethod, ip[4]);

                    Object* arr = helper(arrayClsHnd, (intptr_t)length);
                    LOCAL_VAR(ip[1], OBJECTREF) = ObjectToOBJECTREF(arr);

                    ip += 6;
                    break;
                }
#define LDELEM(dtype,etype)                                                    \
do {                                                                           \
    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);                    \
    if (arrayRef == NULL)                                                      \
        COMPlusThrow(kNullReferenceException);                                 \
                                                                               \
    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);                  \
    uint32_t len = arr->GetNumComponents();                                    \
    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);                        \
    if (idx >= len)                                                            \
        COMPlusThrow(kIndexOutOfRangeException);                               \
                                                                               \
    uint8_t* pData = arr->GetDataPtr();                                        \
    etype* pElem = reinterpret_cast<etype*>(pData + idx * sizeof(etype));      \
                                                                               \
    LOCAL_VAR(ip[1], dtype) = *pElem;                                          \
    ip += 4;                                                                   \
} while (0)
                case INTOP_LDELEM_I1:
                {
                    LDELEM(int32_t, int8_t);
                    break;
                }
                case INTOP_LDELEM_U1:
                {
                    LDELEM(int32_t, uint8_t);
                    break;
                }
                case INTOP_LDELEM_I2:
                {
                    LDELEM(int32_t, int16_t);
                    break;
                }
                case INTOP_LDELEM_U2:
                {
                    LDELEM(int32_t, uint16_t);
                    break;
                }
                case INTOP_LDELEM_I4:
                {
                    LDELEM(int32_t, int32_t);
                    break;
                }
                case INTOP_LDELEM_I8:
                {
                    LDELEM(int64_t, int64_t);
                    break;
                }
                case INTOP_LDELEM_R4:
                {
                    LDELEM(float, float);
                    break;
                }
                case INTOP_LDELEM_R8:
                {
                    LDELEM(double, double);
                    break;
                }
                case INTOP_LDELEM_REF:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    OBJECTREF elemRef = ObjectToOBJECTREF(*(Object**)(pData + idx * sizeof(OBJECTREF)));
                    LOCAL_VAR(ip[1], OBJECTREF) = elemRef;
                    ip += 4;
                    break;
                }
                case INTOP_LDELEM_VT:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    size_t componentSize = arr->GetMethodTable()->GetComponentSize();
                    void* elemAddr = pData + idx * componentSize;
                    MethodTable* pElemMT = arr->GetArrayElementTypeHandle().AsMethodTable();
                    CopyValueClassUnchecked(stack + ip[1], elemAddr, pElemMT);
                    ip += 5;
                    break;
                }
#define STELEM(dtype,etype)                                                    \
do {                                                                           \
    BASEARRAYREF arrayRef = LOCAL_VAR(ip[1], BASEARRAYREF);                    \
    if (arrayRef == NULL)                                                      \
        COMPlusThrow(kNullReferenceException);                                 \
                                                                               \
    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);                  \
    uint32_t len = arr->GetNumComponents();                                    \
    uint32_t idx = (uint32_t)LOCAL_VAR(ip[2], int32_t);                        \
    if (idx >= len)                                                            \
        COMPlusThrow(kIndexOutOfRangeException);                               \
                                                                               \
    uint8_t* pData = arr->GetDataPtr();                                        \
    etype* pElem = reinterpret_cast<etype*>(pData + idx * sizeof(etype));      \
                                                                               \
    *pElem = (etype)LOCAL_VAR(ip[3], dtype);                                   \
    ip += 4;                                                                   \
} while (0)
                case INTOP_STELEM_I1:
                {
                    STELEM(int32_t, int8_t);
                    break;
                }
                case INTOP_STELEM_I2:
                {
                    STELEM(int32_t, int16_t);
                    break;
                }
                case INTOP_STELEM_I4:
                {
                    STELEM(int32_t, int32_t);
                    break;
                }
                case INTOP_STELEM_I8:
                {
                    STELEM(int64_t, int64_t);
                    break;
                }
                case INTOP_STELEM_R4:
                {
                    STELEM(float, float);
                    break;
                }
                case INTOP_STELEM_R8:
                {
                    STELEM(double, double);
                    break;
                }
                case INTOP_STELEM_REF:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[1], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[2], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    OBJECTREF elemRef = LOCAL_VAR(ip[3], OBJECTREF);

                    if (elemRef != NULL)
                    {
                        TypeHandle arrayElemType = arr->GetArrayElementTypeHandle();
                        if (!ObjIsInstanceOf(OBJECTREFToObject(elemRef), arrayElemType.AsMethodTable()))
                            COMPlusThrow(kArrayTypeMismatchException);

                        // ObjIsInstanceOf can trigger GC, so the object references have to be re-fetched
                        arrayRef = LOCAL_VAR(ip[1], BASEARRAYREF);
                        arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                        elemRef = LOCAL_VAR(ip[3], OBJECTREF);
                    }

                    uint8_t* pData = arr->GetDataPtr();
                    SetObjectReferenceUnchecked((OBJECTREF*)(pData + idx * sizeof(OBJECTREF)), elemRef);
                    ip += 4;
                    break;
                }
                case INTOP_STELEM_VT:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[1], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[2], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    size_t elemSize = ip[4];
                    void* elemAddr = pData + idx * elemSize;
                    MethodTable* pMT = (MethodTable*)pMethod->pDataItems[ip[5]];

                    CopyValueClassUnchecked(elemAddr, stack + ip[3], pMT);
                    ip += 6;
                    break;
                }
                case INTOP_STELEM_VT_NOREF:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[1], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[2], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    size_t elemSize = ip[4];
                    void* elemAddr = pData + idx * elemSize;

                    memcpyNoGCRefs(elemAddr, stack + ip[3], elemSize);
                    ip += 5;
                    break;
                }
                case INTOP_LDELEMA:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    size_t elemSize = ip[4];
                    void* elemAddr = pData + idx * elemSize;
                    LOCAL_VAR(ip[1], void*) = elemAddr;
                    ip += 5;
                    break;
                }
                case INTOP_LDELEMA_REF:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    void* elemAddr = pData + idx * sizeof(void*);

                    MethodTable* arrayElemMT = arr->GetArrayElementTypeHandle().AsMethodTable();
                    MethodTable* expectedMT = (MethodTable*)pMethod->pDataItems[ip[4]];
                    if (arrayElemMT != expectedMT)
                    {
                        COMPlusThrow(kArrayTypeMismatchException);
                    }

                    LOCAL_VAR(ip[1], void*) = elemAddr;
                    ip += 5;
                    break;
                }

                case INTOP_LDELEMA_REF_GENERIC:
                {
                    BASEARRAYREF arrayRef = LOCAL_VAR(ip[2], BASEARRAYREF);
                    if (arrayRef == NULL)
                        COMPlusThrow(kNullReferenceException);

                    ArrayBase* arr = (ArrayBase*)OBJECTREFToObject(arrayRef);
                    uint32_t len = arr->GetNumComponents();
                    uint32_t idx = (uint32_t)LOCAL_VAR(ip[3], int32_t);
                    if (idx >= len)
                        COMPlusThrow(kIndexOutOfRangeException);

                    uint8_t* pData = arr->GetDataPtr();
                    void* elemAddr = pData + idx * sizeof(void*);

                    MethodTable* arrayElemMT = arr->GetArrayElementTypeHandle().AsMethodTable();
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[5]];
                    MethodTable* expectedMT = (MethodTable*)DoGenericLookup(LOCAL_VAR(ip[4], void*), pLookup);
                    if (arrayElemMT != expectedMT)
                    {
                        COMPlusThrow(kArrayTypeMismatchException);
                    }

                    LOCAL_VAR(ip[1], void*) = elemAddr;
                    ip += 6;
                    break;
                }

                case INTOP_GENERICLOOKUP:
                {
                    int dreg = ip[1];
                    InterpGenericLookup *pLookup = (InterpGenericLookup*)&pMethod->pDataItems[ip[3]];
                    void* result = DoGenericLookup(LOCAL_VAR(ip[2], void*), pLookup);
                    LOCAL_VAR(dreg, void*) = result;
                    ip += 4;
                    break;
                }

#define COMPARE_EXCHANGE(type)                                          \
do                                                                      \
{                                                                       \
    type* dst = (type*)LOCAL_VAR(ip[2], void*);                         \
    NULL_CHECK(dst);                                                    \
    type newValue = LOCAL_VAR(ip[3], type);                             \
    type comparand = LOCAL_VAR(ip[4], type);                            \
    type old = InterlockedCompareExchangeT(dst, newValue, comparand);   \
    LOCAL_VAR(ip[1], type) = old;                                       \
    ip += 5;                                                            \
} while (0)
                case INTOP_COMPARE_EXCHANGE_I4:
                {
                    COMPARE_EXCHANGE(int32_t);
                    break;
                }

                case INTOP_COMPARE_EXCHANGE_I8:
                {
                    COMPARE_EXCHANGE(int64_t);
                    break;
                }

                case INTOP_CALL_FINALLY:
                {
                    const int32_t* targetIp = ip + ip[1];
                    // Save current execution state for when we return from called method
                    pFrame->ip = ip + 2;

                    // Allocate child frame.
                    {
                        InterpMethodContextFrame *pChildFrame = pFrame->pNext;
                        if (!pChildFrame)
                        {
                            pChildFrame = (InterpMethodContextFrame*)alloca(sizeof(InterpMethodContextFrame));
                            pChildFrame->pNext = NULL;
                            pFrame->pNext = pChildFrame;
                        }
                        // Set the frame to the same values as the caller frame.
                        pChildFrame->ReInit(pFrame, pFrame->startIp, pFrame->pRetVal, pFrame->pStack);
                        pFrame = pChildFrame;
                    }
                    assert (((size_t)pFrame->pStack % INTERP_STACK_ALIGNMENT) == 0);

                    // Set execution state for the new frame
                    ip = targetIp;
                    break;
                }
                case INTOP_LEAVE_FILTER:
                    *(int64_t*)pFrame->pRetVal = LOCAL_VAR(ip[1], int32_t);
                    goto EXIT_FRAME;
                case INTOP_LEAVE_CATCH:
                    *(const int32_t**)pFrame->pRetVal = ip + ip[1];
                    goto EXIT_FRAME;
                case INTOP_THROW_PNSE:
                    COMPlusThrow(kPlatformNotSupportedException);
                    break;
                default:
                    assert(!"Unimplemented or invalid interpreter opcode");
                    break;
            }
        }
        UNINSTALL_UNWIND_AND_CONTINUE_HANDLER;
        UNINSTALL_MANAGED_EXCEPTION_DISPATCHER;
    }
    catch (const ResumeAfterCatchException& ex)
    {
        TADDR resumeSP;
        TADDR resumeIP;
        ex.GetResumeContext(&resumeSP, &resumeIP);
        _ASSERTE(resumeSP != 0 && resumeIP != 0);

        InterpMethodContextFrame* pResumeFrame = (InterpMethodContextFrame*)resumeSP;
        // Unwind the interpreter stack upto the resume frame
        while (pFrame != pResumeFrame)
        {
            assert(pFrame != NULL);
            pThreadContext->frameDataAllocator.PopInfo(pFrame);
            pFrame->ip = 0;
            pFrame = pFrame->pParent;
        }

        // Set the current interpreter context to the resume one and continue execution from there
        ip = (int32_t*)resumeIP;

        stack = pFrame->pStack;
        pMethod = pFrame->startIp->Method;
        assert(pMethod->CheckIntegrity());
        pThreadContext->pStackPointer = pFrame->pStack + pMethod->allocaSize;

        pInterpreterFrame->SetIsFaulting(false);
        goto MAIN_LOOP;
    }

EXIT_FRAME:

    // Interpreter-TODO: Don't run PopInfo on the main return path, Add RET_LOCALLOC instead
    pThreadContext->frameDataAllocator.PopInfo(pFrame);
    if (pFrame->pParent && pFrame->pParent->ip)
    {
        // Return to the main loop after a non-recursive interpreter call
        pFrame->ip = NULL;
        pFrame = pFrame->pParent;
        ip = pFrame->ip;
        stack = pFrame->pStack;
        pMethod = pFrame->startIp->Method;
        assert(pMethod->CheckIntegrity());
        pFrame->ip = NULL;

        pThreadContext->pStackPointer = pFrame->pStack + pMethod->allocaSize;
        goto MAIN_LOOP;
    }

    pThreadContext->pStackPointer = pFrame->pStack;
}

#endif // FEATURE_INTERPRETER
