param (
    [string]$Filename,
    [string]$Targetname
)

function Evaluate-ComplexExpression {
    param (
        [string]$expression
    )

    # Extract the components of the expression
    if ($expression -match '\(0x([0-9A-Fa-f]+)\s*\+\s*\(\(\(0x([0-9A-Fa-f]+)\)\s*&\s*0x([0-9A-Fa-f]+)\)\)\)') {
        $baseValue = [convert]::ToInt64($matches[1], 16)
        $hexValue = [convert]::ToInt64($matches[2], 16)
        $mask = [convert]::ToInt64($matches[3], 16)

        # Perform the bitwise AND operation
        $maskedValue = $hexValue -band $mask

        # Add the base value to the masked value
        $result = $baseValue + $maskedValue

        return $result
    } else {
        throw "$expression - Input string was not in the correct format"
    }
}

# Function to determine if the input is a simple hex value or a complex expression
function Evaluate-Input {
    param (
        [string]$expr
    )

    # Regular expression for a simple hex value
    $hexRegex = '^0x[0-9A-Fa-f]+$'

    # Regular expression for a complex expression
    $complexRegex = '^\(0x[0-9A-Fa-f]+\s*\+\s*\(\(\(0x[0-9A-Fa-f]+\)\s*&\s*0x[0-9A-Fa-f]+\)\)\)$'

    if ($expr -match $hexRegex) {
        # Input is a simple hex value
        return [convert]::ToInt64($expr, 16)
    } elseif ($expr -match $complexRegex) {
        # Input is a complex expression
        return Evaluate-ComplexExpression($expr)
    } else {
        Write-Host "Input string was not in the correct format"
        return 0
    }
}

$ArrayName = "nativeStringResourceArray_$Targetname"
$TableName = "nativeStringResourceTable_$Targetname"

$InStringTable = $false
$InBeginEnd = $false
$ResourceArray = @{}

Get-Content $Filename | ForEach-Object {
    $line = $_.Trim()
    if ($line -match "^STRINGTABLE\s*DISCARDABLE") {
        $InStringTable = $true
    } elseif ($line -eq "BEGIN") {
        $InBeginEnd = $InStringTable
    } elseif ($InBeginEnd -and $line -eq "END") {
        $InBeginEnd = $false
        $InStringTable = $false
    } elseif ($InBeginEnd -and $line -notmatch "^\s*$") {
        $id = $line -replace '\".*', '' -replace '\(HRESULT\)', '' -replace 'L', ''
        $id = $id.Trim()
        $id = Evaluate-Input($id)

        if ($line -match '"([^"]+)"') {
            $content = $matches[1]
            $ResourceArray[$id.ToString("x8")] = $content
        }    
    }
}

Write-Output "// Licensed to the .NET Foundation under one or more agreements."
Write-Output "// The .NET Foundation licenses this file to you under the MIT license."
Write-Output "//"
Write-Output "// This code was generated by processrc.ps1 and is not meant to be modified manually."
Write-Output ""
Write-Output "#include <resourcestring.h>"
Write-Output ""
Write-Output "extern NativeStringResourceTable $TableName;"
Write-Output "const NativeStringResource $ArrayName[] = {"

foreach ($id in $ResourceArray.Keys) {
    $hexId = "{0:x8}" -f $id
    Write-Output "    {0x$hexId,`"$($ResourceArray[$id])`"},"
}

Write-Output "};"
Write-Output ""
Write-Output "NativeStringResourceTable $TableName __attribute__((visibility(`"default`"))) = { $($ResourceArray.Count), $ArrayName };"
