/*
 * Copyright (c) 2024, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.openjdk.bench.java.security;

import org.openjdk.jmh.annotations.*;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.InterruptedException;
import java.security.NoSuchAlgorithmException;
import java.security.KeyPair;
import java.util.Arrays;
import java.util.HexFormat;
import java.util.concurrent.TimeUnit;

@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MICROSECONDS)
@State(Scope.Thread)
@Warmup(iterations = 5, time = 1)
@Measurement(iterations = 5, time = 1)
@Fork(value = 3, jvmArgs = {"--add-opens", "java.base/com.sun.crypto.provider=ALL-UNNAMED"})

public class MLKEMBench {
        @Param({"ML-KEM-512", "ML-KEM-768", "ML-KEM-1024"} )
        private static String algorithm;

        private static final int TestsPerOp = 100;

    @State(Scope.Thread)
    public static class MyState {
        Object[] encapKey512 = new Object[encap512TestCases.length];
        Object[] encapKey768 = new Object[encap768TestCases.length];
        Object[] encapKey1024 = new Object[encap1024TestCases.length];
        Object[] decapKey512 = new Object[decap512TestCases.length];
        Object[] decapKey768 = new Object[decap768TestCases.length];
        Object[] decapKey1024 = new Object[decap1024TestCases.length];
        Object[] decapCiphertext512 = new Object[decap512TestCases.length];
        Object[] decapCiphertext768 = new Object[decap768TestCases.length];
        Object[] decapCiphertext1024 = new Object[decap1024TestCases.length];

        Object ML_KEM_512;
        Object ML_KEM_768;
        Object ML_KEM_1024;

        MethodHandle generateKemKeyPair, encapsulate, decapsulate;

        @Setup(Level.Trial)
        public void setup() throws Throwable, Exception {

            MethodHandles.Lookup lookup = MethodHandles.lookup();
            Class<?> ML_KEM = Class.forName("com.sun.crypto.provider.ML_KEM");
            Class<?> K_PKE_CipherText = null;
            Class<?> ML_KEM_EncapsulationKey = null;
            Class<?> ML_KEM_DecapsulationKey = null;
            Class<?>[] dc = ML_KEM.getDeclaredClasses();
            for (Class<?> aClass : dc) {
                if (aClass.getName().contains("K_PKE_CipherText")) {
                    K_PKE_CipherText = aClass;
                }
                if (aClass.getName().contains("ML_KEM_EncapsulationKey")) {
                    ML_KEM_EncapsulationKey = aClass;
                }
                if (aClass.getName().contains("ML_KEM_DecapsulationKey")) {
                    ML_KEM_DecapsulationKey = aClass;
                }
            }
            if (K_PKE_CipherText == null) {
                throw new Exception("missing K_PKE_CipherText class");
            }
            if (ML_KEM_EncapsulationKey == null) {
                throw new Exception("missing ML_KEM_EncapsulationKey class");
            }
            if (ML_KEM_DecapsulationKey == null) {
                throw new Exception("missing ML_KEM_DecapsulationKey class");
            }

            Constructor<?> EKconstructor =
                    ML_KEM_EncapsulationKey.getDeclaredConstructor(
                    byte[].class);
            EKconstructor.setAccessible(true);

            Constructor<?> DKconstructor =
                    ML_KEM_DecapsulationKey.getDeclaredConstructor(
                    byte[].class);
            DKconstructor.setAccessible(true);

            Constructor<?> CTconstructor =
                    K_PKE_CipherText.getDeclaredConstructor(
                    byte[].class);
            CTconstructor.setAccessible(true);

            Constructor<?> ML_KEMconstructor = ML_KEM.getDeclaredConstructor(
                    String.class);
            ML_KEMconstructor.setAccessible(true);

            Method m = ML_KEM.getDeclaredMethod("generateKemKeyPair",
                    byte[].class, byte[].class);
            m.setAccessible(true);
            generateKemKeyPair = lookup.unreflect(m);

            m = ML_KEM.getDeclaredMethod("encapsulate",
                    ML_KEM_EncapsulationKey, byte[].class);
            m.setAccessible(true);
            encapsulate = lookup.unreflect(m);

            m = ML_KEM.getDeclaredMethod("decapsulate",
                    ML_KEM_DecapsulationKey, K_PKE_CipherText);
            m.setAccessible(true);
            decapsulate = lookup.unreflect(m);

            switch (algorithm) {
                case "ML-KEM-512" -> {
                    ML_KEM_512 = ML_KEMconstructor.newInstance(algorithm);
                    int i = 0;
                    for (EncapsulateTestCase tc : encap512TestCases) {
                        encapKey512[i] = EKconstructor.newInstance(tc.ek);
                        i++;
                    }
                    i = 0;
                    for (DecapsulateTestCase tc : decap512TestCases) {
                        decapKey512[i] = DKconstructor.newInstance(tc.dk);
                        decapCiphertext512[i] = CTconstructor.newInstance(tc.c);
                        i++;
                    }
                }
                case "ML-KEM-768" -> {
                    ML_KEM_768 = ML_KEMconstructor.newInstance(algorithm);
                    int i = 0;
                    for (EncapsulateTestCase tc : encap768TestCases) {
                        encapKey768[i] = EKconstructor.newInstance(tc.ek);
                        i++;
                    }
                    i = 0;
                    for (DecapsulateTestCase tc : decap768TestCases) {
                        decapKey768[i] = DKconstructor.newInstance(tc.dk);
                        decapCiphertext768[i] = CTconstructor.newInstance(tc.c);
                        i++;
                    }
                }
                case "ML-KEM-1024" -> {
                    ML_KEM_1024 = ML_KEMconstructor.newInstance(algorithm);
                    int i = 0;
                    for (EncapsulateTestCase tc : encap1024TestCases) {
                        encapKey1024[i] = EKconstructor.newInstance(tc.ek);
                        i++;
                    }
                    i = 0;
                    for (DecapsulateTestCase tc : decap1024TestCases) {
                        decapKey1024[i] = DKconstructor.newInstance(tc.dk);
                        decapCiphertext1024[i] = CTconstructor.newInstance(tc.c);
                        i++;
                    }
                }
            }
        }
    }

    @Benchmark
    public void keygen(MyState myState) throws Throwable {
        switch (algorithm) {
            case "ML-KEM-512" -> {
                int count = 0;
                while (true) {
                    for (KeygenTestCase tc : keygen512TestCases) {
                        myState.generateKemKeyPair.invoke(myState.ML_KEM_512,
                                tc.d, tc.z);
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-768" -> {
                int count = 0;
                while (true) {
                    for (KeygenTestCase tc : keygen768TestCases) {
                        myState.generateKemKeyPair.invoke(myState.ML_KEM_768,
                                tc.d, tc.z);
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-1024" -> {
                int count = 0;
                while (true) {
                    for (KeygenTestCase tc : keygen1024TestCases) {
                        myState.generateKemKeyPair.invoke(myState.ML_KEM_1024,
                                tc.d, tc.z);
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
        }
    }

    @Benchmark
    public void encapsulate(MyState myState) throws Throwable {
        int i = 0;
        switch (algorithm) {
            case "ML-KEM-512" -> {
                int count = 0;
                while (true) {
                    i = 0;
                    for (EncapsulateTestCase tc : encap512TestCases) {
                        myState.encapsulate.invoke(myState.ML_KEM_512,
                                myState.encapKey512[i], tc.m);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-768" -> {
                int count =  0;
                while (true) {
                    i = 0;
                    for (EncapsulateTestCase tc : encap768TestCases) {
                        myState.encapsulate.invoke(myState.ML_KEM_768,
                                myState.encapKey768[i], tc.m);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-1024" -> {
                int count = 0;
                while (true) {
                    i = 0;
                    for (EncapsulateTestCase tc : encap1024TestCases) {
                        myState.encapsulate.invoke(myState.ML_KEM_1024,
                                myState.encapKey1024[i], tc.m);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
        }
    }

    @Benchmark
    public void decapsulate(MyState myState) throws Throwable {
        int i = 0;
        switch (algorithm) {
            case "ML-KEM-512" -> {
                int count = 0;
                while (true) {
                    i = 0;
                    for (DecapsulateTestCase tc : decap512TestCases) {
                        myState.decapsulate.invoke(myState.ML_KEM_512,
                                myState.decapKey512[i],
                                myState.decapCiphertext512[i]);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-768" -> {
                int count = 0;
                while (true) {
                    i = 0;
                    for (DecapsulateTestCase tc : decap768TestCases) {
                        myState.decapsulate.invoke(myState.ML_KEM_768,
                                myState.decapKey768[i],
                                myState.decapCiphertext768[i]);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
            case "ML-KEM-1024" -> {
                int count = 0;
                while (true) {
                    i = 0;
                    for (DecapsulateTestCase tc : decap1024TestCases) {
                        myState.decapsulate.invoke(myState.ML_KEM_1024,
                                myState.decapKey1024[i],
                                myState.decapCiphertext1024[i]);
                        i++;
                        if (count++ >= TestsPerOp) {
                            return;
                        }
                    }
                }
            }
        }
    }

    static byte[] xeh(String in) {
        return HexFormat.of().parseHex(in);
    }

    record KeygenTestCase(
            byte[] d,
            byte[] z) {
    }

    record EncapsulateTestCase(
            byte[] ek,
            byte[] m) {
    }

    record DecapsulateTestCase(
            byte[] dk,
            byte[] c) {
    }

    static KeygenTestCase[] keygen512TestCases = new KeygenTestCase[] {

            new KeygenTestCase(
                    xeh("796732acba3efdf731bf7c242aeeddf5eba5b131da90e36af23a3bce9c7aa93a"),
                    xeh("91618fe99a8f9420497b246f735b27a019078a9d3ca6b2a001aec0b9e07e680b")
            ),
            new KeygenTestCase(
                    xeh("60d235ddc4f334bfd91d6b7df1a4fed84c88c2933806f13fe06ef15aed96c9e1"),
                    xeh("2026edec2070fd7a2aa93c55d26aa0cda5c16117ccac98cb86d9c99f5bac2cd2")
            ),
            new KeygenTestCase(
                    xeh("c14612e7a22ec88bb5e9dcf865776c37cd5b1c6d1b18798be80b9562ba4752e1"),
                    xeh("3ead7c9457670c2bbf1a3d0f38d6c0838098def05d5589bebee67ff5c33b4374")
            ),
            new KeygenTestCase(
                    xeh("ed5bf4a40e4ce8d367598819be8ec4ed706df4d26819f69729c2acf274515c8e"),
                    xeh("f910824d75a213bc1642bcfa5a6d304e4723ce0a2ca16000e66fccf48ba0a925")
            ),
            new KeygenTestCase(
                    xeh("78981182b43d78c40b828554c36d70b960a02c66490c15a4caa6a7d5f1e9ce34"),
                    xeh("8908c5912b74c4f5d1b3043a95ea3fc16bcb8d85414a2f4fc134f18bbcde262f")
            ),
            new KeygenTestCase(
                    xeh("917a2234587c5969cc1ed10d51b0dcf8b3017143ebf31687930f3e2c610a4850"),
                    xeh("138e93b1f959dbb22c0fad45bcc828b4d4f07ea2fc109eff6adf423d960558eb")
            ),
            new KeygenTestCase(
                    xeh("df022c3c86b725c5f2b54196b7d68684b9fde93be78e38beaeef18195321f4e2"),
                    xeh("71668363f77b645c3278b07c0e97e7a3336421e624485f2ec35f34a01c9189d2")
            ),
            new KeygenTestCase(
                    xeh("d69702e666f4086d18d3da173a6d0b44bbebfac8edad421aab72b823fc63d600"),
                    xeh("bdc7be9d2b2d3d8e7e616fe2f77f8bf11d4759b8247431b0e9220a7f8e5235bc")
            ),
            new KeygenTestCase(
                    xeh("865f638c38f0852d2d712a708ffbd7d96f0df21071d8bfec74c2302ea4c5adba"),
                    xeh("1878346fc47148ba7523463ff794a14438c5b5e8500bb6b92b364f90ec8c3d36")
            ),
            new KeygenTestCase(
                    xeh("741a60c9f1715c42a5a9e67b4e69e5f128372002a6c4f54ae5869500171e2541"),
                    xeh("6078a65f40993ee3bfce7804086962e5726025b779ca7e62912f244b4162a093")
            )
    };


    static KeygenTestCase[] keygen768TestCases = new KeygenTestCase[] {
            new KeygenTestCase(
                    xeh("796732acba3efdf731bf7c242aeeddf5eba5b131da90e36af23a3bce9c7aa93a"),
                    xeh("91618fe99a8f9420497b246f735b27a019078a9d3ca6b2a001aec0b9e07e680b")
            ),
            new KeygenTestCase(
                    xeh("60d235ddc4f334bfd91d6b7df1a4fed84c88c2933806f13fe06ef15aed96c9e1"),
                    xeh("2026edec2070fd7a2aa93c55d26aa0cda5c16117ccac98cb86d9c99f5bac2cd2")
            ),
            new KeygenTestCase(
                    xeh("c14612e7a22ec88bb5e9dcf865776c37cd5b1c6d1b18798be80b9562ba4752e1"),
                    xeh("3ead7c9457670c2bbf1a3d0f38d6c0838098def05d5589bebee67ff5c33b4374")
            ),
            new KeygenTestCase(
                    xeh("ed5bf4a40e4ce8d367598819be8ec4ed706df4d26819f69729c2acf274515c8e"),
                    xeh("f910824d75a213bc1642bcfa5a6d304e4723ce0a2ca16000e66fccf48ba0a925")
            ),
            new KeygenTestCase(
                    xeh("78981182b43d78c40b828554c36d70b960a02c66490c15a4caa6a7d5f1e9ce34"),
                    xeh("8908c5912b74c4f5d1b3043a95ea3fc16bcb8d85414a2f4fc134f18bbcde262f")
            ),
            new KeygenTestCase(
                    xeh("917a2234587c5969cc1ed10d51b0dcf8b3017143ebf31687930f3e2c610a4850"),
                    xeh("138e93b1f959dbb22c0fad45bcc828b4d4f07ea2fc109eff6adf423d960558eb")
            ),
            new KeygenTestCase(
                    xeh("df022c3c86b725c5f2b54196b7d68684b9fde93be78e38beaeef18195321f4e2"),
                    xeh("71668363f77b645c3278b07c0e97e7a3336421e624485f2ec35f34a01c9189d2")
            ),
            new KeygenTestCase(
                    xeh("d69702e666f4086d18d3da173a6d0b44bbebfac8edad421aab72b823fc63d600"),
                    xeh("bdc7be9d2b2d3d8e7e616fe2f77f8bf11d4759b8247431b0e9220a7f8e5235bc")
            ),
            new KeygenTestCase(
                    xeh("865f638c38f0852d2d712a708ffbd7d96f0df21071d8bfec74c2302ea4c5adba"),
                    xeh("1878346fc47148ba7523463ff794a14438c5b5e8500bb6b92b364f90ec8c3d36")
            ),
            new KeygenTestCase(
                    xeh("741a60c9f1715c42a5a9e67b4e69e5f128372002a6c4f54ae5869500171e2541"),
                    xeh("6078a65f40993ee3bfce7804086962e5726025b779ca7e62912f244b4162a093")
            )
    };

    static KeygenTestCase[] keygen1024TestCases = new KeygenTestCase[] {
            new KeygenTestCase(
                    xeh("796732acba3efdf731bf7c242aeeddf5eba5b131da90e36af23a3bce9c7aa93a"),
                    xeh("91618fe99a8f9420497b246f735b27a019078a9d3ca6b2a001aec0b9e07e680b")
            ),
            new KeygenTestCase(
                    xeh("60d235ddc4f334bfd91d6b7df1a4fed84c88c2933806f13fe06ef15aed96c9e1"),
                    xeh("2026edec2070fd7a2aa93c55d26aa0cda5c16117ccac98cb86d9c99f5bac2cd2")
            ),
            new KeygenTestCase(
                    xeh("c14612e7a22ec88bb5e9dcf865776c37cd5b1c6d1b18798be80b9562ba4752e1"),
                    xeh("3ead7c9457670c2bbf1a3d0f38d6c0838098def05d5589bebee67ff5c33b4374")
            ),
            new KeygenTestCase(
                    xeh("ed5bf4a40e4ce8d367598819be8ec4ed706df4d26819f69729c2acf274515c8e"),
                    xeh("f910824d75a213bc1642bcfa5a6d304e4723ce0a2ca16000e66fccf48ba0a925")
            ),
            new KeygenTestCase(
                    xeh("78981182b43d78c40b828554c36d70b960a02c66490c15a4caa6a7d5f1e9ce34"),
                    xeh("8908c5912b74c4f5d1b3043a95ea3fc16bcb8d85414a2f4fc134f18bbcde262f")
            ),
            new KeygenTestCase(
                    xeh("917a2234587c5969cc1ed10d51b0dcf8b3017143ebf31687930f3e2c610a4850"),
                    xeh("138e93b1f959dbb22c0fad45bcc828b4d4f07ea2fc109eff6adf423d960558eb")
            ),
            new KeygenTestCase(
                    xeh("df022c3c86b725c5f2b54196b7d68684b9fde93be78e38beaeef18195321f4e2"),
                    xeh("71668363f77b645c3278b07c0e97e7a3336421e624485f2ec35f34a01c9189d2")
            ),
            new KeygenTestCase(
                    xeh("d69702e666f4086d18d3da173a6d0b44bbebfac8edad421aab72b823fc63d600"),
                    xeh("bdc7be9d2b2d3d8e7e616fe2f77f8bf11d4759b8247431b0e9220a7f8e5235bc")
            ),
            new KeygenTestCase(
                    xeh("865f638c38f0852d2d712a708ffbd7d96f0df21071d8bfec74c2302ea4c5adba"),
                    xeh("1878346fc47148ba7523463ff794a14438c5b5e8500bb6b92b364f90ec8c3d36")
            ),
            new KeygenTestCase(
                    xeh("741a60c9f1715c42a5a9e67b4e69e5f128372002a6c4f54ae5869500171e2541"),
                    xeh("6078a65f40993ee3bfce7804086962e5726025b779ca7e62912f244b4162a093")
            )
    };

    static EncapsulateTestCase[] encap512TestCases = new EncapsulateTestCase[] {
            new EncapsulateTestCase(
                    xeh("""
c6e8499d78b750e516b8a67709468d3fa3b4d74c0a5f5a4b5cb94d8ca92188437040f9312aa41edcf274f0aace043862471591e84c0e05d61031d69f4fa11f4e8a0977673ab30b2bec0bccefe434b55632c6057842d977027ba616e6a7fec7340bb20c7331b4e2f1044e9692ab0136a0b285fde033ffb25543c5c9ffd83595e380f8e831701b1e16e040600c1c7924b6bc42331d987b7d63c007d76d198c0c32aa3e3663ca7606454a214821b15b8ebb018026aa1a618539c00d0a84c871c1baf016bc426b3d9be7598fd8148d63865bd6b455fb8a47562e95666e18405367d4cd581162fb878d8c0c3d627597a4811eee0963d4926861ea202d46a6a6e94e89b3cfdbf436fa2254f060ce6b80b4bdbb98da80417fb53eba8286b58644d8d84654fcae20906863fcaaf89abd992b176b05b2924bbe566728878b5e816b64a532ce2cd41f5a937ddadc15baf23023a19932b337f508c8fae504a3aa24261a9bd20701a2d3ceb3c72ebcd5807d358c2ed45d0b84092d6c93b74caa20ea4b5db66369d26fe64652270949a930bbd0103ea7d117e6160d9dfccded58476b94ae90036b68a40b9bfa61b15c889467a0a01342f843be8227ae9cfb1593c240a8735d8105a36c2b68e5ab97671b63a1cb1eb07495fd3bcb380aae66d20f3bb096b5db9944d83196b3bb3e321dc76c3b7d6828f366585f75a3c6d370f5580f22b385319483c9c19e6e81cc22690b18782c2d47be5c81cbb1c055cad84db9b06b15355697ea23a2374f027a63197b46f28a85d089a120848e94e9838cc72d1aeacf7d64ba69a93635c6a10ca5745645416e071a48056c78a50a0d4a60412966ab4961819918211a661cb839b01101798583934c9ce0aa447170b870306cfa02a16b9316f4f22d9b958a38b24960f0203cf2418495446c558a6ed8706d8c949e7b14cc168c965293d128ca16412ff7b221cad771d441ae3fc822d86645f5e266aaeb1df56a6185747e3312587d1b5f51e13e23586f9f3b1cf393bfb24173d7e27584b061f4258ff3c29d27167f58999b6b87a820b56fc7846cd3a0953aab07bcd256cd997b4e0172cfbdc0c2fef1b5fc616b03d1c3efaaeb7d48c64a0c839085160b4a5732"""),
                    xeh("6FF02E1DC7FD911BEEE0C692C8BD100C3E5C48964D31DF92994218E80664A6CA")
            ),
            new EncapsulateTestCase(
                    xeh("""
1ad51d02065cbbe01bd1025aaaf94bbb2bac27e678fcd7578ef9bf3026700d711aca31cd4a9641a25286b26c034228cd906434c3e2a56f01b54a98878c6842b9bc9c8d95640bd496c0956446a0cd88d344d552a838e29f48850bec6b4667749fc4266ef8d5c47f8b21e7e108f8a58d44394c8711470ce8cfc92b2e49ea5b8b15772e4389564ac8d568b700d11eb5f83d018514d5d3acbd7c63f667ad9340b2dcfb972a333df99ab2dff50f570a4d0001a413a98f4075acc5241b5e6a6132551b9c4c1c22551b5a008df7d5a218783c2810b79a52552afa922de76dc4b89d87761cfabc81d218cc3259457316333c21bc9b650f6383c350e6681c694496565fa7203d05b0caf5a22a4ccb1fe7f00f89297f3dd02915eaa2a881cd41599635abb85890649f1a4710142c4aa4b5e3d71188562ef5205925652ae7606e30dac7099495ec939fc5b0778477be94185c649c858e146cc6750d3a1a98424b2645786681a1b1095979f2313228f96d9daba787a257f4e27a4ed3377dfbc71ccb9ddb227151203798b58f44a280c733a8aa8a631fa303b26c33bae7b088121d917268aba41cc797b7a3336f12101d5ee8900872b3a1710c885811a2d8c4f0a188d4352f259b62da152bf2a022dbb9c913c1c254a44287546c01053bfea0bbe3d05a24f1b70d969feac7c7325ba69ea981c5e179ad564e7ceb4ff38b824f9754af15664ff7a807bc34e62222217b96f364907916be6ef0cc99b014f3527572e0a6be517d2225aa6eb41e4c2a80aef4a212092bdc186598a69b4d0037cfeabfb13b9a9bc0019c5a83ee659100bd76a1da4b30b686ace6291877ab9d834a641abed635b8f3c536bd6394684ba283da1fe7797580924fa6a65e9de87bf833bd2e67c4a9e037aae0b3eb1701bc69add69127a98a5157a6b3f8393cd104c31a24ad3e381997f28c482c5dd6ec24d7747b559a29d56c0f655838fca25c4ad49e6cab35da559b354b55357748edfbca6122601a945d8611863917c48ba20d53b71909746723b4355a620b880cbb63f260ccb10db6d3809a96740ed153dfd07ba60159a3194553fa93ca28513a65b047a7350a0c6f43f7a09f3e2771f461bbba1e3508c8f8931c16ee"""),
                    xeh("4660985A5838041F2E50381CB4E7AC908BAC83CC1E074220C6705E3F5FBFC2EF")
            ),
            new EncapsulateTestCase(
                    xeh("""
335b7558ac204e11a305d8b1526835e65a110d813499acc36a5409a15a8de46633bf587a706a7d388c5099c73adab17035a347b041cb82938da6838a11513c5756b3992ca2a40c91a450505be4cb4fd9625d209110888484033d09f8b60eeaca65330beee240f9861e768ac7218a6e41a1205f55b411e275253706c57b97ea650c6e69876f79bffed8759ce30df61465b4408a18bb003a7aa446642c90ea072f955f46044f4ed7b9a9b82749e56ad470b0d21b67b89cc7ac784f9dd016a9ca5d2b025668421c25a100dbc0328084c2e3367c022b60fc6206e40aab3a863ba8eb1f8682265af69d15aa9d173204533a1a7ad94f693310663212e4aa0e63f812f1608e1f9903f0685c9b9a797ea5939869b5421b87dfa1ce818265d5051038a9331168ce01971efc06cf02a59c9e1a03373a600f216f779aa73171be66f95d81808daf63a26cc3709ab58f22659ca9587c01029d915c7554946b217044e65a6c98258f6a0a2608a9cdb4c35a0e06c93b19988c221c96b79729cc227ad5c3d4f558062ca3179b8a0b6c0d6de0aaa466c498065ce690278b8980c7c567d308b181b06667f297ee9caa28ac2f5ab454b8045eea583d20993bf69359be500247b3a2ff0b95e243af49b6a08eb19c00f31f3611837cc64b85d8bc35da9466fc12967b158dba95782ab771c134f9945df0b02644fbc949cc633419c48643764e0679a690cdf0239b3aa49a58e171241a106c151b75197f7432cca3d6c133cca0a4902ed6c299a479224ec7baf4243152877adf3255ea114fa01a65a53ca47e3a6dff072d991730414754c831b3763a6673e9b8f4f7ca4d3891a33c542380cfbfe16a650ab749d79e08570629116ef7d50338f26834e298dec8abbcf588f37b428f211434666a1c6041a3a09918293fbff54d35c75c7dba2720a0a91592af264b446105bc5b41344067bb3e5943d62aaca4374f92933e3d4353f1898868d097e504cbb934cf727a9861169582bc4414e46c50c41693909cec50beeda69aadd6073fc1ca3e1a6a822751a404730c3c90a4ba6a9f4343988280eca4c4600b413cfacb8ce2823f2a2b96f56d9d1effc2d3292c0720e7bfe6aabcd8284abd8c3fe8aaee1bfca3"""),
                    xeh("0D643FF311D83CEDCB3A95BA0F76216A49BCA389A225396F708EC9A51BF18517")
            ),
            new EncapsulateTestCase(
                    xeh("""
ae76c67fc7215e493c8645af1c58bec08b66f66a92346b92f873cbd646aa5dc2b9b6e4c2848a4324a8318e9983932055c4f58f42346df1da07977101434a9378d1a1559819756646eef3094b5ba55e33b596b71c04d249be4b8318a4becb008ed71000ae795ed1ac873e244beb7c4071047e83a8b3c6c2be2ab025726063175c4bab391597923d030b5a9238b131383d15dc2a4bb812a8539f9b0b0876a23ae049777e6229c1068a37ab13a2406c6f7ba4591520d8816ca3894bf3c601fdf493e70c5f4b19bead520d25cb6e8439a548f35cafe81497b297341005fffa18b99754cf199ec90504f9f16f8e415cba718318437aeac50ba8413e540b963cca2202a35e356701e177ab0dbacfe6a0c02b506fdd6436982737f903b8a8204cbb07c380340c5f1a84685392a2b62620004caea2c416e61f7c7b0e6e202d92021628cb4c10d57d5b60ba4d771712d552d1f9c6f352779f8c32fed54a8eec59f4b3177c7c3e81e3ab7dc83a396bbe38ac0fa3ccb174442ce638c6a0fc1fc9c426b5eab83d0397b559a97256b2b0817ee115b1e026c445629bf7a8803ff71aad99385a1abfdb22a3be0029503cb44b7a35847bc675f17437d9246cd104bd8608da3005e14a632d69006ed4434c80594b2909fe284203571eac04a00a120c24d050f40952fd53ac0cb39bca50c28ff450a6e208fb26bcacc1ccfa89bd7e05616a395aafd2bfa59cb541975cbe429919b32cae4b1f4a4b4b97858390c164fde76bcdb32b54b3325392c385d647b265578a1c56dfe5a252144ff1c02dc2054bb53525ce4b3bc2ec190e3ba06fbba12f06707f060bb1659607971f3726b9f59191e6e7b252602f51eb281f454f1d80a60ea29eab42752df9a3ce376467f694fcd288c9fc7851c04821b53e693208867604def746f787ada2097740ec8462ac89baf84c0d50bedb539ccab33894408c5b4b91def7955923a283677867a356b78c5a8eec308bfc982c8bbdc9e40d7695b42612ac27f6c118b5379518b49e338738e830fb6a66488b7115d24895963934688d91fa9608d41582e79e5d417ab07c81d39415fdbb870a3f503f1a197bbda81d7c3204a316392fb75016f88565a73a51537728ac7011"""),
                    xeh("AA28DCC71FA83D9997DD733D8B0D0394D84D33A3D3E1B74CB74DC6049628F861")
            ),
            new EncapsulateTestCase(
                    xeh("""
59851028c33af8acbc9ff37ab19a9fbe45a0f08b9cdc2268d6e727665429f16448ed840d0a87c34c3422ed857c69f3168820b0ae998702ac6f55150ae3031a54fb80912ba94fb4bb31604492f406f654855b447b39d26bc81b0d1644c1ce2a7f347c239cecb0cf89ab14917192c12e7a694581021e96fb6868282df0a4850bf3acfde5424781c571e8857cec1f3735a464ec74d859b169142e2e7413d2a02c5e93570e3b9bd18983809129724743a1979794a2500e04a409506ecf0a975f58bb0d08b36c1733927c932e7c73afc47f81b5a87f25878d327d0c375c38d2940aaa543f46b1198345fce7b7bdecad6ad22e12155bc470448f98bac8ebb96bb34b552ab3bfe5255c6783a76425b386613a112ec1dbadbf105ae0ec3cad61c7032bbe0bcca9d3eabf03ac6f0d4c5e500348f6581876e80c531197e62071a9454947b450614761ec95bb7072ab53e81d89e47eb875263ae74ca6a28b2a2a1fd44947756c1f3eb38519c9b2d6d63197e6459bd1427f992680b1a6f1480cdb998006a25cf434346d04942732c5617245033b430b4ca726b94faee24ca54919f0842233e1c6268b4bca31be4ca7be9b52851e2ac0dee573f7959eb896b0bb18064cbb9b71585a319c74b7716de398a8111214ae3130bac5b196e6734d3cca4116bbf67ca42492be22f56b538a8532c6acf4404069a329735246e5eccc2730c2f5690c53510bb2685909015874a8125178c7fc00709b2a6720391e38a0193aa96715f13ed628068ff4bba6e74c7be095a897ba3dc50f88b2b08a353cb4c4bf382a5cd72227cb671ed74227ce245268c38874f53299604c41f15978c522e865333dd0c49cea97289813a828ca76b93dfd3987da9726f0d3492cf13ea72ab8d4e82e73a3987b336f0cd2157de0b9b4209869412b9074736867b10fa82b34e5424c5998295b6a54793e9ab119e9c68049ea2dbef55e27327080c6ae71d69374b8314ce6110369b76fa363dfc201729a700528944f2bc23c464eacf51519f0aed8160d5ceb66d10c865604b3b4ab05f1c2990bf2b5ed84898b6a891ba360af171b051b907d1245770038e379cb727784b219a3c5ce49fe6273600219bcc609c8ffcdf4acbadcb9"""),
                    xeh("A4BAA4C603DA1368C1F2AC552A331F77BF1D598C6BCB540D43CA1E6D4B8BDE77")
            ),
            new EncapsulateTestCase(
                    xeh("""
861927e1d996dacb9c04c630b694150f23aa51f26bed8c744c36a7916864b5160e94cc515ae0a03ffb445245c56a434f9351ab64f6bcbb973718ebb612657d64f844df3b85bd484c473a685a0a4a94e9995d865aa5bcafb7dcbc7e9877d63890a86252c691044df94ef7a19c68056ed6315400f8bd38f48abcd09e82f0376816c0aae53a9cc61b466466b63938dbb0b1dd963adae40802216701d094c716211718943b543cbe0888b1dc463c20804c4c402105130dd7a87a48c3c3328f76a8aef2b0c7d22688e7636b88d62897809eadf1ccf2dcb5a0011998099506925dd3b74869f931394bcae43466a49a40b31075f074bda69c0ee2f7bafbf81712969b43f0beacc246b3e71c9df2c47db405e235ae97e1a99c0365076b29e593b39c485521716967751025bba279f3aab85c6bae3800f255396d136459545f192b650d3b142c383669d2547cc2732015bc4645554638284642b918b5bd65a06fb49c5fd267a90b77ac35d12da3512f44d2556e027e6582529cea194df88584190480d9b0b5335fdfa8685894514cf95386cc1cb63131dcd1022f5b568cc621a729423c0567a51313f647918368012a83816a2caf88e062a9a6064fb52d8b31abc270a01d974459b4134be74c41b6a22e72ba1c993924bc43be6509235298431c72943bbceed41a84f9b84ce21abcea516f32280a0586069acee7627b54055515604a9ba112e620783aa76264ab8136759af9dcc75718a9bac22092a645b1b5cf32a92e377ba03a6117ee842172505793f973f84cbf0fa22d3ce116ef4c1012bc69db94c7b2b461d978c27c93044f22972588af42e28dc3748299e7a098b5566d67110a036ca9c6740dac84205109f05104f80585970a7906d7bf93067472a34442780f0fd25729745c7f85198a753836c3381f0592f818916b48278e14b61f1616c32ccb987940550316462257f1daba0574223ed6767b9c696b1c01ed75484c1738cf60afaaf9b3144899c4dc00fdab36fbd792dbd4885ab3af6ed63f8394b66d0a910041b1d606923858432f81c37f57c146db8dba6775ba1899d5904f3a69abee80050d6232d858d95a7bd86ee6af48bf9c372170c7468fc024cfdd552fb21ae23b08"""),
                    xeh("C08584D2F5C950E371668A4FC8F527E20AF1532CC28EE6B5620729155B06389F")
            ),
            new EncapsulateTestCase(
                    xeh("""
ae13099bb82ebe29ad44859860634d62f78fd6b26295d040f8a5c7ff7aa393d127e214aac9107ccc06d029da6a6b998736710c5fc3a6c9ecbdf00069a9eb2942b45b1626b7953886248caa59789f9e988fa26073bff9b703232859463a9335374506ac57a8c2936701fcfc0235d071bf9807e907734e6267d130c9d4b2406e999b6cd15adf7783db2309598c058a93ac043319194ba89857a59eb35fe4a3cd6fa1c9d7d46ba2f6c62ba85ca46cc8d450bfb0170f5ad6a2a0388ebe73a8b58627f6cc5e7477635ce32472a87b4b881ca9fc749ecba012187a5bdc6e9e12b668f32786cbb5fa4055f86590e00bc4340c04b8b0053b189a74d6bb22db2f75f6b32918c0c3ac7da62044db4a9c1c2b0b1ee81b5317b1fd734c33fa3733d7b37bfa36c16578cdb63cc6f73d661a35e6460b74396ec6a07111091071a757d962384940a1a2338a42fa68e81587fca304d8a86282230470c8988cd707437163cd46cb489b010d235fcb976e50b51e0a22c5662800e74903b403c025bc6df9d821752107b2e51874147645687fedb108c1894e0ac3801bc1398c4a187bea7d9949bc5cfb8368b296b2fbcf2b59c71321b5aa2055e5823f86113a1519c956d9c1793a5d52a906a5031e905a862fec5abfa0965fc49a2db189eb6bc69cb8882b703e0ab35e088340fc35812bc0a184956fdb448170a6b32b49ac0e2a65a1041d38d4512dea3fa111c7e428cdf52144ae0708c5c44382a0733de356ab556e0aa141d58cb1321058be25a543d7060252c9e87997c35295596c02bc7b926f5ba05675a5dfea5a3eb62b2521854f21498c428a23048fa38484935399bf07c64fb63f7ee793a63aceb6842bf05077d2b3647714332d259576182f0d9c508cf2321f0090a6c1cb0a5c106647101277a7b1248e74189df6b90c26a612b26555d433bac49a3ebd702e6d9649038233d6da01f77534a8d2038f615a502cca8526c314c5856a033ec97155ad79a2e2a32cc1e734cedbc6c0b94219d13bf985588400cc682294bf21183830b2bb51b87fd9073b9cb6237884df0cbb12b929f81448e5578928b07193ea4141d19b4eed07a956c34b6de59127c5e823bb8f6719ddeaf282650cfda0e26e57"""),
                    xeh("1D51A0CC52E85972001B77047D97DF5F47AE11FFC6C31B4AF42FB0791A3DB40F")
            ),
            new EncapsulateTestCase(
                    xeh("""
d0f903b9113e3b18442e9b125dd762cc4bc2fde8007fca65b4817ef985667bd62bd8806563846ddf61b68a837d35859344e96ac450cc24303d9f116d502135369305a8cc8549641487f88e0f24a31c87cea974547abb569f9c312dac3fab517c85a5521d41b7eac40c62b7cb8a0776d89828e937177785a23ba0599381004c253fc90aa8332172f7366e6c756e6f266f2c5c9bc70a5cf18416c00c83dbe76a8a9c52ec5929217c0d2be88c8e3374c2111c2ac46a2a454ceab38f8d368f2281749d4109e90702bf368d6ba6ce212146d7a21a9ce6767fdba3bf1708fc359fe4e974b676bb9ee04db18b93f60759a071484c134e0ae29b135c5f2546562f6a471a826aaa3568bcfc721d707ccaf9351462098fcb21b441c81f876a7782363ef84eb9c59b2eac7f637ac205f0c1ddab4bc0868f42c4c2ffd0912a0578e9b81dfdb400a6a747c9bc8c97ba3ff8074f454367a5768d48653cad01c30212997553941780b44f3b26b9b79e6bc43cd56683dc9770a1d81ec0601e66326148f8645f820737c2479533a7cc3142341a2fba851cd304423d835bba5c73c3fc617f393e673591040876e670b94585902818883608a773e736f52c496691481b908106609366513ae14b618b8611539b56b3f466f13a5f59e5179fd29ac4ec7a92b5beb004b628e7701b11681cc61fd088a031d80d8e43b36042857155b0c5f207260bbbd30146d83a9c3c17a1bd804b56c96b01e71de75152f4b4990a001a1d8c736e39097771b819e58a2bf4498f1ca346cc851efb29c1551060998a7242c7fb5b04b6c8cf2f703056f18a9df28504752be88860507b1aa517914b278b61704f2e33cb02307a7a898e58a3a87221898cb57243000249709f5e53037728bacc7027dda1b7fbeb7739375811427b3f8c3dbcc0775442323289bd89ea0b668cb458da346dd25a45d4c9d6eb6ad98645a65794a795277480382ad7a99e85ae9e279ff5d38ac9a461d4a8aa2170136e69bb70c2c7734060421388330606763158f0c04788d79ded7b52b79abdbf891c2290b561ab4a6aa02e591637cdac7353798d9e256fd0286389f7d3b079e3e9f89ec0a68010202ca26dd2ea953f551554098ebaea56f782c3"""),
                    xeh("BC2D661E6283B835BAEE160D1448957AC2366DCD087176E252F81F1D11E28781")
            ),
            new EncapsulateTestCase(
                    xeh("""
ed8034619a429573756962129b459916f158f0ecb0cae396f60c4c84670ce3fbbe119055f1a4382cf2a619c31cee5842ee8c9c6979193ba82ce9f94337306186b20a276c6d06c04f41a414c3100b31e2a2a9984067fac4a7b3c70747c55b87b2512219d62c8b51a6b628766d63d1096f63bec8806311415919336bf9d1b038022d00d1bb84387888dc10f0290b752627c2d78ca5c8475150bd925240ed94af40b43762a99a3f5b0f41d702db7cb485405d5a247ff48635ba7719aeb8a876f0986f233212c40a3645c3e4ea3b5fd02bb520b9d8aa22c4595849b8a2769b92edabb4ba1528dd067aa86b3bdb6a39985983c89c9c916b41b34018bda438e7815bdd4316f6202c0e3c73d53124ee110ee596a9e3198f24b814dfd77f0a2a7aaab3a918b64c47fa44ff018332557cd13325a8081e0002096e68a293258928896532ac5b0ef4bc28bb43f68505f8a61cde0b5ec9ca8d62c261b5290e16b22a4a978d5b141240043c9b457a89a764d9565067cba987a23d5d6b3016291fddac50c9e5581d988088a368d1c9c633bc594d39c56fc5b4a996b7c110208de1394559915138b7cd8941daccc7382b60b47770272a56f846b15c71704b8b7e08595d51658a10b348617c8b552588a20a07c6e0670416636ba0982726671d796803320e0346bea80578f7808419f14b8be67f8bbb627fe4c5c28074e1458ba909581f317e70a66320ebc5ffdab20202143f374742b77d88c80fa8b9593d74a04e2c7113101400c4afb7893f634565aeac9eb8c144a3a172a1263fbee499d7895e298b195d4caf46514927d39f430aaf13c4bc43735670e5b0350a42506621dfd737177ba3736026d314675947a1b7315844da146af84c4cfa8227bc3211895773082fa22a8c9e7c20d1257ee9577c3f36a6d3097212dc1b149421b9bab30124cb3298a6b1505bf586070f52aab80a71e17c223184b32e9a6aa4627b3b99bf089973ab53be7c461ad57317edc3b79872bc50a391a43bb18ee9bf0ca100f16409caf4abbe0239b98b16a598ce86e7c07b27646de1671e915763ec8dc3c09a20181e0525c571a18d74c74c0e798d11d22db509a0f0519c7440e6787efacb39fb35c8cab4a38302c0"""),
                    xeh("6745F4F0730AE3F14A428A95C9CDFE82717EAA94F65B00A01566A4DCC9ED1E5E")
            ),
            new EncapsulateTestCase(
                    xeh("""
e782aff3f25dfd582912a662095a71ab2b81a7b88a0a0b531d0c735e84279a39713e9b78a39679cf2aa956a0bac8a8b8d6330eafe18a20890492b42ff0eaa210a97f931c8c66a4bde05ba317aa5a7c1bc4e7ea9c8a2aafeaea6216d03dacb9b1e17c30c49a58a5d21a05f37912dc336b8739abf4cb74455078655b7d6644f3d5b91f3a97b9317676156dba77747b0cb2ca21c87e7cb0e3238d67811f4e229ccce2303d358164e85a5ba766f91212a8540e800a6e5b829bd097925a01ad99f8af2b8c7832640c0e64cacf5cb5449829ef992bed3a85b3d86fe8689de552acd1b47f58b883e718c61b810a02563484e63a951927210cb5b1e740c1bc2132d026bbb8a9b2099922ea44c2445bb078cbd0accac1c22c0065b99601c07cca67ced147645508a50ba8e7a0012394cec77880a4916c653803266c0635741fe0c98812f37265842e6826ae18d4b27a334246520607dba4452ab7284300f96a01ce809cb4011df18940c6460c7a33aa34e997b89846a287af180ba68f170653e065c1475f48b9763d6bc2643269631729a11aa33aa32a14525641e4a68a61967326698be67d0fd2612daa94dfa9a5bbd3b87bf57ef75639e3c859ba8c0a90f5ac131c94a0644eae3c32811335401ca4c0994df4711c30ca05ca812667d47a8f96beb0a270f75ace7cea1047d4abc944cf71042aa092a041fb60d60976b0ec120600c79c51ce163a6e11e1b972e703ceb5c792482b01b7c617d31922f760e7465fa427ca118431727bce77c12c59b2a7cda287bfbba68c57aef92a4775c6602c87330d693bd10a5d21724ab58bcf03220a14c4463c5226b86a1fe15477e662ca03374287da5b7aa118de1255f574220d0bcc4f68b7d36286761c9ad5d9aebe422b26e95f1f5b67bc012950572374fb8e19575b2a7b911bdc8130d737d4e34ae721c76d131baa9a2f47e208c668215fc2c56f80b16a75114363c3a04918c0072d61f19d22e55d9d1cbfc9210246e12a530ab7c8a7633ca37b20789125573f866b94d872af914007c3ccbe898ba697f63d15b57b2aaa10145c5bd71966d54bb8728b531c87892711e6c0f2f1b554cadd3cfba90def490f706c74c451248631333f0b3f2813fa"""),
                    xeh("C3ED79224CB07A8D37DC9C789BC7AC8E278968E429087E5B2C0E878934DAA53F")
            )
    };

    static EncapsulateTestCase[] encap768TestCases = new EncapsulateTestCase[] {
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("2CE74AD291133518FE60C7DF5D251B9D82ADD48462FF505C6E547E949E6B6BF7")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("76D04F481E68B2F901ECAB58B6369A2CC31A9DCCED82A1BBD426BE0AEE266AEE")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("FD3C91294D8C974930B4B6135AB647D4A7885C83FCDCB30CBD38332E14094491")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("7DB18CA35A53AB3A65E4C17FA096DDECB19FC7747E657B49D1C1710DBD1D197B")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("876B17263B409171B746C6936EC65FC94137F958DC974BF98110A1D07F6D95F9")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("E0AAD46FDDE0B8E64361C3233263D8A751F5583DBE91AAA6E69E6318FC7A8EE0")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("90347D478D5D964D66A54BE930FD9F7FD3C2AE1492DAC35A6CBDD02616BCE14A")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("119BC36B5F856C0A2F136B3EE42041B817125A600E829FF6B4B402131A26ABF1")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("697CC7445AE2C9ECCA2569B7871F0BBB364E63E4B782F734FAFED4FE33E4AF14")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("52CEBDECF06579F4A9351F77CA95B5CEDD034D812F3FB7FB50320CA80E4118D5")
            )
    };

    static EncapsulateTestCase[] encap1024TestCases = new EncapsulateTestCase[] {
            new EncapsulateTestCase(
                    xeh("""
4f31c22cb34ef4a51478c635302525fc0abfd221ca8b3a6e224338bb0b9116ba54cd581cb7a3813c878d65317d8736881c8a6b6a7030441b44dc6663a6919c7a3574463b706105c332693d1918682d2a0f2e12b1a545a8aca6a38f0b63a216463099558d282dee3c33af3bc160d48b889a9d80e9cb6d83786dd826cf2c58c23c87f0279359026458040ae638c3eb5c5e0f7b6ff2e76ea598bac5143941d50d031a4906bcbc826a364d92641f68cd1b43973fe9b802f778963515cdd47eba329fe6b50c8bc5bf6d3aa02e69965693997afbae7e2c8f06b85435403c827a39d63c8cbc675d4fc31a5a699a0f2381918448a4471eb5aabb24c0685e6a1ce042553c360190ec470b44a7cebc153186be0d8c8996a9b3bf46383aac698758ca7c9b4a993b77e8397cd3a3bfd11b31b961c90875c900157f97e211ee6c853fea2dd83212c3f8861c0a98963820d6d73c7b3693b738519deca589639462db956bca22513b9cab7459a51296443076ff732e3eba57ca767d4374a8b2544c46c54c2f46cd7bc6461b8105f0d41ca4a532c6e4052629929c03bbd0b3c6b52c51cf3195eeea6aa9a55de0d9063bc57722a72409a59747965b4e6a5e14ec112a103bba204977d43822d20c92a2aa0f4882798216cd26a857c6a12bc78ffddab1d73a9b2f91ba4de83bfc54092d4359f03a4416230cd6616dcbd916b989856f3b0672b172e20732be0730d1a3a6a837c59a05b846ec1d600841594b306acc67cce4ab7b9715afe074a75738aed749c1494dd683ab6791522023acd8e5c403f77b23f128f16261a100497e0b1f906b7956c53c1f1a0d65d9422d7a3e234c252675446f085450446393c0c151e751ce25cc05b12a58c334ee1334d16b813c7689c6851b7662084ef05ae30bbebb5c2911859db6dc449165bf8b41626839cb37814768167ebd24300306af259941c7628997bc80156c6b87c758d6c40b55076b74b2b9942093dfc979ae59ca16a6ae33b523a7c687f3f22638f37c69c4946a26bdede88477c7b67538115cc890873ba223a953e1b79f7a902d0b20cda992a25b235f30164312111a01cc5d82d4a6e1c4961ad96bf48b797680bd0d04c83463bf92d86279c994c88b57f5a14c45645f28fc19ee835f0ba480bb04a17f5853c1e00c37f0857e88504166a230344bbd3180e6421432874fb6767d0a0c52c661bff33988d3e3132cb744df5b6c28ec4e7ba2b25c22a179914f043bb93536602d754c92cb6ebe42832f605d6dc99a8065cda1ecbdc589a93ed321c529c5f910ad86b9183b827e1e8005e0ac102a35b5abe533e31acf35e649f967c064963675421990082f4897971ee4204e3611bb3bca7af3a5507b239e9ccd49c5c4e09c14f3a3106bd961d9d649617212832c54f430140c8590dfb1bdd40785390384bc9640e2c8564c4a4f700ac1c6d1895fa40ca309aa5054b04934bfc190a45f78ac1a8811046710ded3246ef4032f3b73231913db1a58f45563e3a62413f6a0598264321b33460578f682ba40d29ce15697d6373bc6220c0931bcfe87326f019c7c746e0136a98a6bbe8c7caaea8a222e80915926c4fca60dad75a3f03a60375c09b5c5b120180b51f185157a345dbb91bb5aa6b704183ea0ac0f374a90c02206668224f975ed605ea1b01c55b9a2c159c2a096b3c4a98c2cb8b81682864188cceb3bafe825016381b8b2bb7df2ab4cf2a71f77347eb84314dda92f6d5a5c34d9cf3a123a16b892099c23dfdcb5b8e67be1087ec5b38b081289fa8c3d8d560a2d882f8b8501b4096d82602e7960a2c433abfe928e769b3e22965cf7f7350a1cb4543c52c8f3983dc84709c1909cb76084771c3a59867208c1bdf869b85548740c8b4390a2a7824d2b8a283606aa04a78b79acbeac33a2e4490a44b2995c70165aec4e37a79ee3224a8c3693056b67b1021c5c473e7fd31551a24a3383741d082366398b835389d4e4a34f7905a4c5808b27b44e86aaf02032ed07ab30452bead1907ac28a1a98a97683895ee583ec59316673213f9b1e5c2339678c0b6770459509ba5478830bba25239c311231b796c8225c9206facc138cbc2f58495f106b019041387982cbc656604e05c06e630b3d84aff82240b7a7cf6c078b9fa99fa426c197a5aaff94a93ebcb61de47f9c1c368b4f0104a0e8d4f20f98b38f6962ea4d6e053a96a8059c6e564a9ebc6c"""),
                    xeh("59C5154C04AE43AAFF32700F081700389D54BEC4C37C088B1C53F66212B12C72")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("2E2C821791D3EA49D0AF380B97AA24532F6109D85360A751BB8B4C048C48D26F")
            ),
            new EncapsulateTestCase(
                    xeh("""
0799b60055b072088af992055fd18766111d70a92dcaa1881c1b5d03807e4a948d0937b9ed4385a6479a23930878102c0c9a04cd1b51bac6a7a81cbae47b15f33a331a5303695a4ab229585d27b3a7105419952bde5c659e089d38789d3a82212e03cd0ed7a6efaa913362b55f386d2e4b8d2565a5781642d8e970f6c9ca25d2acdb8a3da7f493e37331d26432c530bb401a4ce65c463d24194b2b1d708ab2a34009cdb02b1a96a921f42c90488940e7033090bffb30b17b06bf5f3b7115a24baa94a1620434247ac6c9579f42294e1f50cbba44829640bb7713c8ee957174a85b295bb1e44943db036773d91b211537fd9584954726cb83c4498658d1d2a10d2b6b1593348423aad4f6bcf1807d6599a24c4544dff617a2710e119bc17785b0a8c57428ec993e5991f76a9503fc580e4c62a0b296dc52a1a2c83846b7673ca308fc0877d5f473012c5c50a67813424792171e87a7ca555c0bb380ac351baa05074abbecb602610f1c095a4d3aca0bfa11092bb8dac09918449149960dd7d0928dfb2fc23b7b50d85012416e2d237afd47348378c024ac5b14c9ab4c5507a1355685550de7dace35cbbf20a679c9b76fef5b795e0a14c803c36984669cc88acd814d9380646edb1cfee159040c8366e7396af8977673400f5ab2c21980e2a184a472563b1c32886c58a0d68a2d39275529c3fe1a744c877a0605c5da7556dedb89ae8731e729b6911a6194d71102fb42961bbbf770873784af83c596a66c2a94a7c851d9677c779a0ec527bd761dec9bc9d1b22645eab949e31f1968c649c811884a765ae00e98f3228fb03643f47890f78fa6f0369dc37c98b593934a3937084f0664b10c091e7a1315adf4b3f34b5b20123905009c6f24c7f0c601b8f53664b6bd6d48cb487b0d22d45e00f19409a894f3ea32b2b3ad0536ac05e420a74a710dc8375aa9610d2640fc7818ed599598f60e21113670b99cc0d96447f6069e0b193ae79277cb29883506180743678b3c9b53718974210f043140d74392d9be2c08bcf796b462211d60d618faa152e461cebc3aa675ea1260f4585e810cd4ea8415687d3b2405ca72a46129c95f9bafde87061158920c4b13a3c5300918c8de4b0f840410ee91077d611971a4094f3a236230337127c5a789209f36c39b05b6618512ea7c9549aa4552d30ebb5226bf06708b4621a5cc8f58b8153ddca32748682e36363df63bc1249ce1205484048fc8798ba7929fa627404f072a2e0c02abeb48ac9475dc1c087e427309c2b701668a291a54056047def6b10618a34c550235aa6d39f05af809174041b2140640c252b604e81e56f34f2f711ce6d64e56ca0fce0133244481058aa477628022f2c2eb6c544d1045dc61aabeb0aa42127bda479160f582cb075619f810ca2ab886319bf87b12ab710db03b4a1a170be0699f7f333a46b18d71e6832c356fcef57db82430b1bb745f2215fab44571d4b0d9d568fa5cccc2919b2ff6b8d9281e07759069b9a602ba7642f570b20b15a156a1c322447323a4e9fa1219b812b03a5b2e1abe8c5b7ac26a8589733a3fd33480e18ac200362e5a0dfadb1252c0b2d8c139e4532507329ebc3910f05b480ed196bf86166bbb2f4a96a91053b511c81ffb748f420a33a6cb771dd0129cdca037e4533624aa16fc8f18a9bccbb475ef0a8b978ac9090a4539ca1fe06c5501f2b34480b025732a90a01e60ba9dfeabb4e8c61ed5c4396423af0aa79b86157e687c313e5623e6137f67a25680eb6bb841b80530362bd1425ee986b18830509bae9bd9926f020cd02713c7a83e8ca4318a044691637d03386dd28c3ba081481ca7649939bab47668fcd4291a4885317025e5250894e968168447f9f5c73702a5837802cc4593ab99242a59a39fa14935eba3e781a7f2740fbfca7141c821c8921e7c52b4ebe24732118cf57590dc919d09ea18d2ca63a605495bc2a88f9922eb356fe0343e4ac2666c49adaeea898b1c023058bb5bc4437bb6b798f59ecc8b565f811c8ffaa13cf8a4af8374c3221c3e1ab89cd94393053acceb40fd6a2c1d2c4c6316993835b9ade3a4bb396a5040574005ae9f1619c2fa66ecb363be8a7fb424927f8033ef8b2397a691d2670195590e9299375ca9b5dcfacee6eabe078b28e7aa1fa314cf3c866667cfcbbb5b57f2db74fb19e5e0ab5683efda06a93631032380727b6a80"""),
                    xeh("5729B2AF60A4A5EE3BA6D7F255D7D2437812579942FF2C6F48611669135DD695")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("FE8AD6E3F3EF1FD1890FB7FF75A8CD9B2A04CAFA7ACEAA99D06D116B81039DEE")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("0AA3B1F8FFA63F89F949DA18B6D8570BC5811F85A4BFB293E9D411ACD43C3227")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("2429F93D29E48EB6A25ABBA3EE2F3423CDDDD0ECF4B2090C6CA5BF4883F4F3BA")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("B65043CD3672CF9AE2CACC94F923CEF63B5127ABC63C2A5AE6C064B8C6FE7C57")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("6C8C075658F4257D42010EDFB1D7EA290D3344EE6E4C43DA799366985AD52243")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("BD990171C3252230BE21FA7F186A121686187B77C234C37CA5122A7AC77E318B")
            ),
            new EncapsulateTestCase(
                    xeh("""
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"""),
                    xeh("135056EAAD8A28DEB1BE77EEA30CDEBC7B3DD89D1444DBAE145F39898256ADB3")
            )
    };

    static DecapsulateTestCase[] decap512TestCases = new DecapsulateTestCase[] {
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
ea231b200f36afe0ba1bbd719812ce994d5f6f2e60403c1e4fc1a295423e38a77d66bb06645c4ebd004542d2e39655a1bcc461af8b5d1882b8405902c8c3f63082d29f263cfa9527f04d8ac9eb99b9ed322079c5bd62c8b29c8a22992eee4cd6d7152da756a405bb0565808f3ee3d592c659b6da737b7ebe5f7196b859b04dafaaa10056488d4d455a939cd640a0031dcb9b4e5dd89bd634ad9c8003b924f76702987b6914630c5da035e34ddf40af1bd6fee7eafb0c602d91ad848f45ed222ce1334bfeb6f69a01bc4e5a11d044a421adb30dcb08d86460592f2b61f67ff0f02165b1340e9f6e93f88fa8d9bf1b82da7f21fcdddae82117881416f9b48833cabf7656d138e9dbcb29e0f4d15f11f303227a8ee6d04de284d9c76e0f35103ab4303e67ed76ca2f11c2d440e6df881c39fa1f2fb435b6f8c517c6a341b1cdf45064e7b67c1cad43b2c171e5557bb78c5ddeb9af3ad756451c70f10a60157750d2f40a5ab9fae21621484b110e7232b9ea9c52ed2c1331045d555dfe87ebe3ea13c52966080fc140728a26f4438fd25b4e5559a9ecd0c3a73e23c976abf2cd38a1e2f61e6c17672b569c08f3c2ae49add14370a5c9bdd3e5e6a4c0f7b3369f95dc33c30cda832b349af1f4b8699ceb1e17d3fcf515c2358db02b9cf0ec8b0706675e5003db6cf2182e34dd9d030e90620567fb83d83df293092efe9f0e935d79bda625826d45cd4fba547005b9e416c07112286eff057429dbee4822b935b3e16f9655a5fe6f362c58c7f52c5cbc75b733911378f174f706e4bc0b37848d274a29d9de894f5dcca67f6b80185b371f3b37e43d56e374070d6831f0ae48ae96c15ca80bea4eb6f37fb528064f09eab83f95d40c4a80feba2b1860b6c4bbf44bf3e4d4a72a28755f00126f0e4cf78a81cb1d8ed56f615e043e7b4a4f219c4f0735f68589152b5e0a81241ed434313635924eaddb815d0495d5fe065bcf5c6870b76781e486643a858d972e35d9c61be9f517f28963cf836eecb05f24777f600962ece48ab25cd32b80b9e96924f7ecbc1e3f29e011b086d9627770670fb7ebdab4c5""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
cb7eb96dc952919624e5825a24ca80f9e2a313bbfa357af50438fa22870029c8f876cbc9cef450d124df902e93618d8dbc8d1344cc51fdfcb19788872aba1f13f673c387dd64009e1ef98ed34dd5c29d8849f503dd064424b39a8b55bcb714e6d09877fd5d7d1d721b2b32b871cb453b1b00587038f5c313b807b356bbf7fb9a50a3573a2af83f0e893c32a20cd95b9046cac886f74967e6c73e718b0cf8470312cf1a9241bf25238a3bb54e01b471bfc398b5e23ea7cce32ade3169dc472b25fc7cb68997f2bee116c86415245beb30a01841acac6bd06c82a6644aefa24601a2738fb1bfc2b08199d1574dd87d9b220d12866199c01b6c3372c204855ce8884ac7393da5479818f26dacecb32e43fc0d9affa881b4952aa4b95b8cc563e1cb4b06ac2edd783bf8cc4be2dcd0d3105f73c593f40771bba6432c654f34849f704ed007652628db269bdcc6ac14f1a15fabfbeb7529a5f3d70244fa2e7bfb4efa23199b88defc15821581f6b3ccbcf3fd75e3b41dc8a50bf21309e395d8e616d7655bb3e99e3c4cf09e192725ee5eef558f84fbaecf144c69742fdb40aee854b95c80878904773311d832cc5673ff0078c4f877d4f208a538ad243ffe87b4698af6ab0d207b338715518fd794930a8650fd5cea71f035623fd19dafcfadea4f53db42e76f5e61c46f1da934a497c3142ceaaa1d2b514b9149f34d1bff9c5f71206959b6e24a034e5be92da9713d79bd885c6500ff3dab4082d488e4932420cbd395a5d612245e81e8824d317aacefa6a4c93b82f2d99865dfd5d8a03623b463f0deadc1345d123b8ff876198cfbf945c0366a20a0bf6d5ca3814504591dfd731d033a30e2728b74f859f0b91c9a16bc310450ebd69d4da09931fdaca4b5721ef2627053a47b8a290018ad50a4c2a0e543b130402a323bd94bbae64a175790da350128e99ac8d9636ba12f5de00582a61bbca9acee2eea84be49dbbbdd730e0c078f94af4bf44ff8a8de89003867f94bbfa55bbd905d7a260c23ee448c198fc7a89787ce879c04a7a77f379bd55dd8c33c484b9dd070b116645a4a44506b8488db""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
7ac45cc6b87f431f05a28b0b6f8ec44fa020f8a0f798e4463e084e58c950d643a72fbdcb2a8e442aa2a9efa2345111838129097d942773073581f8828baaafee93d3bee80ed21e13fab0585b329de6348acebdf25e09f13c1edf96789a3e6c7b7e1143e0a5f408e33699c68ba140410e738e9501e4a087390970a86e212e3ccc18c62b4233ab94f3cb7faa3a5a8839d68514d851555704ae8a601d7e1df4f3e31390905945fdd88f0eb8a39afade14b0ebbaa16077acb4e3ffa41931bb057e386297d90045add3ff509df9650d86d2709d89a5fef748b15e5fdb8aa5c2dedd84533b5ba49bbe88ad78224379fefa0b68bedd76379b5b7a06235673f83871201316d9f8f050645f6d8031fd951ffb2fbbe4a66f0010abc8752ec509ea62cbf2a59c3a927918b06ae3524b6d94765f5ad6856f5ddb2cfc036f2520368a4e15cac62089adecca6b83e8b9896b48bb4aa4f33656c7a2ac3c02539c0525ae71481aa87be9b956ef8b6558f0bd2aa62d3be2489a4c477c605220ae63ce5b67b3d517eb53085a747c0566449c06c9a54993a2ab089ceb3183d5e69173b635d52b87d6c1e0dfd4c3f8a15e0cf2d2cccfb2d12429c6c174aed9d59ded4bed6ffb964e8d97d4d10ce0f2cb9124a4f88fb864bcfb6ac358824bf0e7408e19829fe038b481a4b161ef06b584eb720884865abea18a05a00d45b27ac0175ee1667f6feae675d8a4b85c2a181a52cbafd73d5e70becd0ad0b27b08b14ec2185e902bbadbd3c7a6c1d49a3a8e65568e590b44d12d6dff7abebd9158ce588d73a387f8dca572b51c24649e2ff02bdfaa779759b93a70f8d0e302a25b528e26a038298437619e4978f1af582d19a11ea772fd0e705dbcc92129e4e9f88857b0cde3da2055427dd8565b20e1531f701361b731bb2dc093f259a01941e255a1191718098e5477e27f7a3188339a1237e742d71882376b72f55bcc4ecbf03f1b29e79a1f0b969619755fa449f1727635d9eb71b977f4acbd58e5c72e89433689d5a2c759f7feb2051eecb6e4e2d3f37e36d863b4aa86d78df37b4dfcb453cae1c7869a0e67f6af60d90f""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
574c9a1db78f7ae6ccfcc6ef48b8246db140f1a0d94a5bfccbe8b807e08889de2b45666e78b96eb159a73916947c2ae930c389b6c858d52144ca1f5c6d66a43f6ddff5c233493592e89f196220ad4360503ace196ef8057800c2d05b5b12e98203986a45c0dccb4ce1bf84b76e3f6a84f6e8356956f0b05c02ef2552e135d90c61264c63033848651c71464108be23aa0004fa1ffac1f4e1022dc915abb7a9f66b8770386e4330d7a94722fe54411fe0388599e6ca1463c9ff6401d9b37d42a3039232b62a7f8dd7ea125a4e54feedea008d51eeda575b4ee42031c908ec52a4bdf61f1ebb051c65cb9f9c4780958f2fbc0fefd38574e7b09f48412a7990246cb64dd8da41d68f0c16468dae921d548240d14b3023c830cbdd130aa7aaa87861f58f2530f9052e84ade036b717b365c8cd897b20166859c95bd90a28ec0316a0784a9d30da3fc1a80e012ecef3d1b12f91766fde1bb4a99f4cb00a94fed0f345870881e777a6852700068f5d321be5311ac813cfb3f5acecc6ed2e0225e016856ca379bb61806e3dd5c264c31190a7f903f0691b92a55ecc40a0a683c09636842df5337b6bb1b2616ddafdf9e526b757bc33831172722d95cffb2162941ddc584e70055b1fe21eadd8ddcd04c2d574ded14fe924c0990ba81c2eb6e146e483d87655f139fc651a1070420de9d3d82e9af7591a981d1bbf650682332f3bec1ee2c291696af40d54289598327ce64c32c2c77a81a6b4818839e8a6145a9eda06570ce5a348a72e8c332015387298eed54daaef0d4950f1c94646e1bc80547209d23d51d50fc86f4fb4e32351ea2b7a6451a8caf36e211df0a7427a46d9ce462335545f133fc6864286ae2349ab61d1a993da4f81ef523b3f1ffc6141742aebf63ad97cd6b78df8d8f73b69a4d6efcfbba2c44da1f1bda3cb41e725221edcb4a7754b9d79fd8ea8bd2a11c73d39c9d10a3abd5af6c145964601a0478d9c18573907fa2d88ec51b56ffca7b51a7a604706fc035dfc12366430234a46863c2fc03f3fb599b7c99b58231c28bc571bc629192fd3f15092bbf5718e2644a76bef5af0a3""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
2c3f3d5b5ab52392b227104e8b9c85303e6d579df35c1e7849ebb92e80ecf200ca9d9b9c408d9f68b4f4e5b0875047edb2facbd6a100c1e5a1465df22a88dc1055a663cb2084409d33ffab500c4e20ba796115a2a3f23228542ae90ede48e05dde69197b9d8138b6359d1aaf85443cea446a4a6b0fe769d2ac9de4a6eb1a4f0d307c6277a3fb1e4022667233b90885ad45eea2f6dbaa54d737886bb504c61b53fb24772c6c7efa12a7b13ade4d6c29e410a604fc5ad87f378e703ce9f709f9a7485aa18bf7e0b4d7d1b88968e06f085ae69d49fdd22c40533b07dbd7642444dcaf8e31c5c7c15f6261799663c5b00099261b747635a091c6759e2c1562a10adfaf75d120c33ade725b7cd6fc0edb63862831dbe6e73e356b729cd001289028596900b714f72d47da69306e3e5e823198ca4b45f56e1f60aeefb0827c44e4c16acdc892319b6ff3dfe138305df6bf7c518aa48241a1927171fcd1576f3279cc727fd8c6296b919296dd165ccba9544a8aa01a54df914c7e910d2930e27f7af36ab21929799b4ed216b971a801c11bc1381c205c55a76b9efbca88ee3ed2eca5f3771cce86617b5fbc411afd6d5d2c6e2a73675dbfb26cc7dd2cdf8a1f50ff8644dd1b173e090c8a70184bd14836254b29a489375d1b1a896c3a991a0cbaf86c93c581d0bb922acdac1bda28d3520bda5e4ae8815a72f1afcfd1168c4a4f8c526b12f9cf201e3b146779b2a2ecd35c56bceff5c0986402ac5783e1bee5b044cf6fbd94cdd7dce828b2421addb9bc80cc4aa298b2a95498c165ce241dd64aac62b1f17798d439c46ba24ec1290434e45b2560e8ddb938244c6557b52ce5f2d73d2d6ba3ae46bbd4afdcdf948c27d05b4476cd4fdb879d495c8405abd5ad4691f3853fd95253a22c4e012fea078e3e967ce666a5ebd6492a9fa2019be09923437d59b4ec38dfa71664020ec591a53ab5a2d6d77b3d44da2ba20a098a445182f62171b482cd5583ee290226bb7aa3eba414fc14dd7f0bb56e7b647269d92258a4e272b68cd42eac155f000f936ee48cb625b122c7237bbadfa4359abd3e82ae9899c5""")
            ),
            new DecapsulateTestCase(
                    xeh("""
96527bfd017c70facf0511955eb04e1c8b6ba35116b6c7242af57a8efb7045577eb101087062518f9421cf2b3f09b681ec2b8793b973c338ac4d39505e296d027cb413981ead738e27202782095589945028b5540dfb3cf120a22cf347c3e225391750c9f93b9799a0367418017164472a762c63a8c209b44260cf9997cd5b33be79fa2dddf177de723644f094f10a95920c2b27ec31e78b0521faad545c2964513fe8839014e9862dcc18c1cb088096b82980a1dc586c44b7610a6a90b10861a081acab7381955410f74970cac1afb52403a0ba7f81177ec65b3e045816361050b4226950a7b1bc474093d9c1df124773c82300778f52e8746a54808826399bb1cb7d7026340283f34baa768b785f62b449b9681193419af289c1eb845f16af7bf709ab3c7d00507f62457d74fa048942556d3877e54332be622274ca89c298b64a46c4594030a031c795523928f11f708ccfcc2b75db1814e087842596a575b35079184b8dd61543a94e93762b5116855a423201b66fedf028e25cc9a3d2733eda6b364c30d3c358fbfc4e563a52be95592c7a1169267fde07436c31250aa22285d229b04a1cef106c15b79681a955dda7028b3c205c188a32081b780500d1cb41df8601b7fc976dd67f9810c9d7b809ec3a725739609cd573c3038095235560e9989a06aa910486da5a24ff041071675f0ac327c6b4154f892f13230b05d37ff7f1867c7908f20a461839864a677d818a81d0b7c871b64de4082ab3d472503bb21535a34145bd2fc664208328745b9a06959340f939b39b2c02239db20204964b9329c3330104bf81aa127f75cbbd1822d96c99aa75a8562aab2a2ca9b5fac442988465232ce1cab05fb9c6abdb4337201a16374b75069bc893913ee28dab8b2e56d309ac2c0d244122ff755ed68474e3c254ffb03dac4c9e73392444daa7f8e7658780a49b44b134750eec8b4780c308587287c8c8c726041e2e70cf6488735a6ca6d300956fa1c027c3679e0ca96a457a1beb8e60f139eb0869f61872fa2c130117c321aa073935131a8c9be7c800e6270e71822492c992bcd26df43478861927e1d996dacb9c04c630b694150f23aa51f26bed8c744c36a7916864b5160e94cc515ae0a03ffb445245c56a434f9351ab64f6bcbb973718ebb612657d64f844df3b85bd484c473a685a0a4a94e9995d865aa5bcafb7dcbc7e9877d63890a86252c691044df94ef7a19c68056ed6315400f8bd38f48abcd09e82f0376816c0aae53a9cc61b466466b63938dbb0b1dd963adae40802216701d094c716211718943b543cbe0888b1dc463c20804c4c402105130dd7a87a48c3c3328f76a8aef2b0c7d22688e7636b88d62897809eadf1ccf2dcb5a0011998099506925dd3b74869f931394bcae43466a49a40b31075f074bda69c0ee2f7bafbf81712969b43f0beacc246b3e71c9df2c47db405e235ae97e1a99c0365076b29e593b39c485521716967751025bba279f3aab85c6bae3800f255396d136459545f192b650d3b142c383669d2547cc2732015bc4645554638284642b918b5bd65a06fb49c5fd267a90b77ac35d12da3512f44d2556e027e6582529cea194df88584190480d9b0b5335fdfa8685894514cf95386cc1cb63131dcd1022f5b568cc621a729423c0567a51313f647918368012a83816a2caf88e062a9a6064fb52d8b31abc270a01d974459b4134be74c41b6a22e72ba1c993924bc43be6509235298431c72943bbceed41a84f9b84ce21abcea516f32280a0586069acee7627b54055515604a9ba112e620783aa76264ab8136759af9dcc75718a9bac22092a645b1b5cf32a92e377ba03a6117ee842172505793f973f84cbf0fa22d3ce116ef4c1012bc69db94c7b2b461d978c27c93044f22972588af42e28dc3748299e7a098b5566d67110a036ca9c6740dac84205109f05104f80585970a7906d7bf93067472a34442780f0fd25729745c7f85198a753836c3381f0592f818916b48278e14b61f1616c32ccb987940550316462257f1daba0574223ed6767b9c696b1c01ed75484c1738cf60afaaf9b3144899c4dc00fdab36fbd792dbd4885ab3af6ed63f8394b66d0a910041b1d606923858432f81c37f57c146db8dba6775ba1899d5904f3a69abee80050d6232d858d95a7bd86ee6af48bf9c372170c7468fc024cfdd552fb21ae23b08c1e38d8ce7232e254e153d4b2c4fa1fbf93c919bdfa47bface7a3a8708917b45917a2234587c5969cc1ed10d51b0dcf8b3017143ebf31687930f3e2c610a4850"""),
                    xeh("""
ab66f3111bf51917775e0b0ddb50f0b1440256896320369968c564019c6f5236f4b951cd4370965a16f62bb2046f67584e4b3c2a6e048071d4c69835076b2bdb5f1c5323ddb937a371a418b0dd84c21ba037c2d94470a279b92e0ac53dac8de04f84ef1632fde456999bd2b596242dfca9484fde789794808c8cc5e6cb80ba87a850ecbee0262440402cd7f4ff1c6ba24879305b4f3542a7bb45c767c78af8d167a5233e42f0ac11d1c3a09aefca29eb25b3870fda30d97c53f988ed4820a7b979d3bb5aa4a3e09a962f98c58e3a2b231e6932fbf68398ee649642268950e7c3538268f02553caa443e6e3e29a32b2123603ea7e194122f0ecdf02f206c95a90c3605978b871ae70710887f46ecdcbe326368d909a062875f3beb750b366884802a0f6dae1ada16e55507ac15431c36ab1e01d54c453936d6857b78ce0384d970439aaa4e422441b1f8706eef30b6e8de22fbeaecd3204f8e9f40f1a6ea0825da1e0c967e3dcedb62ddd1befb3eb1c98d7decbafdf326268b79714b57dd7cd0f753c7619bc0f6ebbc8c4be474a6fbaf3d739c952b7f2ef8522583cea9f0a2690349422aa8b09f55cff3579546b2f3371c8592b459e3c0eabf4f871ada231c2933a6b19dd6c1267498d185f2d6971c1a73b4d61ba9eeb4797fbdaa11b87798e3caa78ad03e5edec0ce389c551855ae967849ada410e996b8cbdfba75cf9eb311220ee294585700d1cd15305a5c9a5925fcc6e46d8b94225fa01b69d70f0666466f69d58b0d3f1466d0972ef697ca0b8a40e19bb14aa83a7348338962882e05a79ae341b622ade01b19239e85ad3ea04d6d5d7d9e0d2db8ddd892b7ee6cb832b8b0911d85ea297e169f8d9623e12b9ec50d6e20a5f5f2a5607159f0ee44563851bd229d05276d87ec69aae43b8246e26d1c634fa5595e68a2c0d8c6f480f8642bb62ff80cdb0337f8f4aed0d12f773dfda04871ca37988b86f2b35c0babf83e45797e5b9edd97112bf079c470dfcd10ad989fb871cc66c696bbab7cd1c76dedb2c136192b6730d07a036fd1e3d176d69313bd3ae595f8b1ad63fb9439f0a53ab41""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
697f53fe475ffe22de8a70436eb11099547e47df00dbb25b9ccf2c3ba3a48f8876694671779a72d10510bd703ab015a425ae50d413e9dc5e87548d70b84cb124e4950c1d546249c8c8d60fd06513bbbf6648230538bc5e88269ca0a961b9c8186abc70286289d58b96df28c39f158c1a707f75dd20371c63ff038d0ad388a22449ec4fd8f73fc82c75f1e3e6e1c9f510f3ad8ca3124aa47edf0cf150f7a664a4bd38bc73dc94a722b878b1ea51db19260f17120fbb99916e596ced350107d43bb26beca35978db929df046400f8094391830c99cf2d5cd0e1527f97419550ad1d1c31f94cbb95d3cc5ba06bdeeb973ee20b7961feee5b4c05e08b0320ea8b3be528b52bce8750fabccd59fa7f8bc9bb8f475e25f1884ce4255d4191863e20156318276d5fa1e7d2385c2744ef31c9950ab071f031d78cf28ac07c54f997ae8f07915b16b8490f1cc028ae6b4ebbe42e5d7b5d988abecd8ac22bae7887658f7af5c488d91993c72df113a6b7f435bdc3d90cb32af64d6dd617e5d0b0f611ec2a0575b347c1cd89f92c72bfa3e7ef1e3e3be4bf68ae05fef6da532ff76c9e9d7f7d354b2e638803509ec7757c78c35dee3c1149d2b25f8d18a8b6ba7b03a519d0ec7f18f895d575fe3c557d8dba332e2454281b676a96b3eb0d38ae2de00893649e1764e2cfe0b1d08ab1d69cf65049d3a2b7ba894910c6fea7eb94e64ff8a71c64ed4c9b6b4ad71597adefcddf0904364c50084bc6687affe770ddbf0e8e9268a64e70a9481d6065d2a93072daaa9583d9b2a20db31cb7d5edd37c59e4fd607ba897dd5afbc2a6403af20deccb1d4b3abeb9502c40b5440e5e9a7f850a38633bd5fa75bacadbaa330e7a401978a7d10c9332daf141580f8317fbe7c06da656e7942ea7d5d49a5bc28ffd15d17cd0dc337f7fe2f20d37ee71b51bb536f4692588acca8a3fc498de1d23c9a1b95c7efbdbca3874e91658cdb7755f5bd51553e1747737b1ce2c44604987e3c04d819d7449b6a40fbeef7b3cf7bef8df39926830c558591de535b8c5d482101a5c6a0715ec13910a243ed0d64efc997d2d2342032b8""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
e31362859b08dfcc50ad8f9dba1bfab3249b8e0b15347ae776b41dcdfaaf8596cdd6ad0eef2098b4759f8b168178d75f5a3d13c2e9af1c94d3e8598b779f14a5951928874d86a95b757816a5d74e086f81bc3313e72a1ecb637cac0212e57b3eb756056663ba1a1b066557816a8bc3c73615193bfb16543901948f7e0932eff974bd678be5002cbf91320d055e6f6245a6e3f384d88a70f0916dad162cd4b959894366032e0cc949e1e2b32a660717ef5cccb890df1357698c5f19afbe9742b87b0a077a67d2fab90488813517a8c4e35e2c58384e66d8076594ae713fcebf4c590c209c2eb015c24ade0eb37c341c922eda2220cf09a278aa4c63f97d317cf86bdb6129af96432fd53fdbf305cbd2d7ed9439760cd297d54d10cf7cb22434d51ffc04babf2b04b5acdc1aff426e9eb91e604149c8311b42ffaf14136322d8678e3fb5542423716ff4750c37c2e64fd4b772723b8c67c5876e3d99f29c8aacc118dd9998d11a0a7a18a29d593cfb1ec9a4ffa16efae2ecb4f9fed432edad543cdeb00025a734cd7f7b4c7ac96901468050ae981ac5846a6b7840ae674f97ceb5fd22efbb1ce9a003693883b84034eaaf39e87aa2ab39e782f8984b32ff7ee9cc832181da7589ffcf52562519c63ad3aadcbc04f479924b4ffff5d04fd683bc7f0c0612ae7b1d711edd39dd543efe54d87e6232005b2d9bde0ecf7ec6c0139da18161fff97055fad31f9d8952d370e25990b7322c8ee1ff5d4766e5c3ffa8b036aa0233d031787c5a6d4e788e0793f5bca432934029921014209478c6cfa5b4aa77311edb604fb3e794b61bb1a1a8e58ae4a537404df32a7ef520a8cfaed3cf45a1ed7d08059425c9bee99120d9f528c603acfadab24170c8f6cba3fd01a223324a63b5345a6def27bd5f27686deb78d0bf673f434201b9b1c24658e3f6b7f6fe2110e3cf4d8ed16c02ac9005ce57652c3307fc86f533469bd972a6d90ce8b256a9fdc42da4dee12be96954e71fc3f2a5f8192a97a7a22067bc2503cb51c5ab63e7b6f909166615b407a29b8cc8a2e9792f77836899b8f120852a53df8d4b3df6""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
e31362859b08dfcc50ad8f9dba1bfab3249b8e0b15347ae776b41dcdfaaf8596cdd6ad0eef2098b4759f8b168178d75f5a3d13c2e9af1c94d3e8598b779f14a5951928874d86a95b757816a5d74e086f81bc3313e72a1ecb637cac0212e57b3eb756056663ba1a1b066557816a8bc3c73615193bfb16543901948f7e0932eff974bd678be5002cbf91320d055e6f6245a6e3f384d88a70f0916dad162cd4b959894366032e0cc949e1e2b32a660717ef5cccb890df1357698c5f19afbe9742b87b0a077a67d2fab90488813517a8c4e35e2c58384e66d8076594ae713fcebf4c590c209c2eb015c24ade0eb37c341c922eda2220cf09a278aa4c63f97d317cf86bdb6129af96432fd53fdbf305cbd2d7ed9439760cd297d54d10cf7cb22434d51ffc04babf2b04b5acdc1aff426e9eb91e604149c8311b42ffaf14136322d8678e3fb5542423716ff4750c37c2e64fd4b772723b8c67c5876e3d99f29c8aacc118dd9998d11a0a7a18a29d593cfb1ec9a4ffa16efae2ecb4f9fed432edad543cdeb00025a734cd7f7b4c7ac96901468050ae981ac5846a6b7840ae674f97ceb5fd22efbb1ce9a003693883b84034eaaf39e87aa2ab39e782f8984b32ff7ee9cc832181da7589ffcf52562519c63ad3aadcbc04f479924b4ffff5d04fd683bc7f0c0612ae7b1d711edd39dd543efe54d87e6232005b2d9bde0ecf7ec6c0139da18161fff97055fad31f9d8952d370e25990b7322c8ee1ff5d4766e5c3ffa8b036aa0233d031787c5a6d4e788e0793f5bca432934029921014209478c6cfa5b4aa77311edb604fb3e794b61bb1a1a8e58ae4a537404df32a7ef520a8cfaed3cf45a1ed7d08059425c9bee99120d9f528c603acfadab24170c8f6cba3fd01a223324a63b5345a6def27bd5f27686deb78d0bf673f434201b9b1c24658e3f6b7f6fe2110e3cf4d8ed16c02ac9005ce57652c3307fc86f533469bd972a6d90ce8b256a9fdc42da4dee12be96954e71fc3f2a5f8192a97a7a22067bc2503cb51c5ab63e7b6f909166615b407a29b8cc8a2e9792f77836899b8f120852a53df8d4b3df6""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
a93e00395f1aa0795a82b149b5c8ad8dd0d51c577b2b394adc282b8482a3cc3a8cb8aa2bba8a40bc799074e5b30d17b567895533cb28fbb11c05e07da2c8e080ecaaed70fd5ab2c15b33b32de50b2072a04c084b4740893c7caaf5b5fd037f91b75149cd8ce00512c4fc12f5a67eca00523e00dc7d4e60908189983848e2f529ef5688ee182ab82c44e4f5e57b3a37e6599d34fe299705de154ad6ed4e5d424ac6843925d4df9b5de48688397a39dd487ab020f38d7dbfb8a021e099d9e87aea4531c3bad076c0726667b3aae460a8bad9c5597dc49105abf2d82992bc73638322f3399d86cc6dcf5a5a8646e7100426d6f587b688d8196c8df3ac141878561d144ce12a49ac8292fa309a9b0910a23781f686e8b4915b96ec71372a9f0645caa8e0fb5e4a3a67f465f000aedbf2923bc9af8cb5682fa2fbce090600f4534bd3e384829f9caa5b9ad9c100078071256bb1bc0c93267b4a1885283a4e3d9157d48d19dc97203f19a28a9454ef99536f8a669dfe4432051950fd911939ed2e4e28d46a78d1395dd2acddc031f3218ddd4441bd5be01830277d11927e5d32d7f45494e3c3319b2516b2fb1dd7945ab16cc8227b43d356a12fe6f8c5e4e36752aeb900a02109427cd384ae6aca3869b3be2a90021137deb1f1b9abbfa0ee4fb9b4da554d83b37e0d9ad6cc28d2190d2e8f24e0c26e716270b1ba0312ace2632922a2d5d5343efbd2c37b5b4e8d1463cb8c925bcf9cc90b65625eb91e723e8a9284d165392db21b1b1d2430f06632dc0a3b87182997613b94508954cc95863a1b93808ec0c384abbb3bbb829ea6ac74864b2b5e75907d3ccfe077e461981ea59b78d7d4d521ea2bc68f47bbe403d7419876486d56a28ec5f84526e45745a6fb23cd61b3558f0fcfb408ea6963b8eeeb4ba095d30fcb9fbb81235fee1b0bafafd718e602b5a2343fd6c884b3c084ce9f1cb310f8b9811dbd94778ecba5db61e3ebdae372763e64a0e5543e949d0fd6d4b5e4a165a2ce597c704bbe6c632fb312361a211242271d8a5fc2eb638ddcaaa43586388b98087221ad59a924c5fcacbfcf9176""")
            )
    };

    static DecapsulateTestCase[] decap768TestCases = new DecapsulateTestCase[] {
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
9059c2e3a96472cd2e7bd86c3e78574ea36e2586c919b94a386dd22c9a816b3c9e70842584959abdebc1e208ea779851efa48a70258fc3f7d76c327d536e0f253f06c33698c8af6bf3eab1c4835e5ad76bef5da6e23a6f14f5d47d047a4a2b71ce626207e1f5d6619a992ba98205c6d1e27ba4ea52d450099988ad9a2800f3a18af4d4349ed0b7c04bb4a76b7c7f339898749aa56b8a6d274931904eb1e75e1f497c1ee28abd951b34fcc0dbc5885ebf91bc6eafe79d2286e69416c128ab3915124f80b6c19159064327433105fa13d97ff255b672fa51b6ad3722967f413dbb13a02a9014f6755c89559c40a7c6e9b61f4c48559529f1f9413824a2e466ad3dc3c8086f113edd0160bcf00574087db6b7288b1ad665781412086a2847f24286a11aa37f39088ad167344d0591ef98b458b75f5d8a4410fcfaa83d613166f7fa41ed17cf17ce4673f9cfda45592242d6f36df8f658dc6825d2ccc30d24be4cc553ac655e64477ff91fc70a26a7fe53a6ec2cdacc75872a7a2e6a30387443db17c694bbd9578971f1319e636439cffce1f46aad2f34d60ce2a8d96186b48e64240d1d0a5af878e7636e15664905f7cbceea895d97610cdcb4767c9288e0f8fb0e059bea8363f5cc036347eb2ff8652c68acd16ef5e0a77724449f8a143b23533209b050d3187e5a3ef08c9c5dc11d9836b6396f15d4218c98bd5b440853ae5ab0a921d476b0b143ba87d355b5bde28f751695c9a392bd2e29cd60e15d5133f0a636bc05c014cd623ce9c1ba5ffd69470767a9b779b03bd00971ea534ba1c5bcc0f66d3db1aa731566fe3049850aac69a461f648df35d295f91438ff2ea8b74de1937e29582b9abf960c3c74d41b63955348822c31d350b03de0b50e4233f8bbb3e3acdb2eeb1a6b5c68cc89816ad7266277a8068a8633af85f7faffb9d94efdfb61360d1960393f71050fb4a1fe964377860a0877d18f6449833f02228bcaf8b4eeb2bfb79fea35a4a117b6e5cd8c68262696fc3a011a64b5c73a524221c38b42f6e8219f418044419a08f670fded4634ab006403d85e8d03ed1a0d1550dc5ca41271f63fbf2b9361b5e196f765a18e35dfabbdb2a4b4b6ae7ce176a3a88eddde4591e5ed0936ba88e5be7b05862cdeff319dc9a4ec0f3446d8fd6545c3a240689872b9e92486a1837bec167ac6bad6dacd3e114b7da845b86a4b9a92ec3d7e37327012caba3dd22b8c4c2d2fc1c69f4d6ae6913f966444de9915d3d4a0185d82999277690ce9fbb9187436d8317460428cfb8e5869b9c9a48ec7f3d7c8e16b7d3c1111a5fa6b4f5ad80e5f2cd6c0a331afe0c554c01c61057bead42283e9f115689948129065b38ef077cb246c83a6e5156bc14c134a470ea44aba6c6f6d6b38138d0ada2ea680af00ca499801470beb646f820a427512146c60a857f812f7cbd89989d5a13311851baaa0deb66e7e2bd5f03edb21a9dce53d486434538ac17ce82e7c1ca87a39e31e039159b6f9b1318b5fa4122c3a7af4aee297d2ee2f6219""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            )
    };

    static DecapsulateTestCase[] decap1024TestCases = new DecapsulateTestCase[] {
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            ),
            new DecapsulateTestCase(
                    xeh("""
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"""),
                    xeh("""
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""")
            )
    };
}
