%feature("docstring") OT::SmolyakExperiment
R"RAW(Smolyak experiment.

Parameters
----------
experiments : list of :class:`~openturns.WeightedExperiment`
    List of :math:`n` marginal experiments of the Smolyak experiment.
    Each marginal experiment must have dimension 1.
level : int
    Level value :math:`\ell \geq 1`.

Notes
-----
The Smolyak design of experiments (DOE) is based on a collection
of marginal multidimensional elementary designs of experiments.
Compared to the :class:`~openturns.TensorProductExperiment`, the
Smolyak experiment has a significantly lower number of nodes [petras2003]_.
This implementation uses the combination technique ([gerstner1998]_ page 215).
Smolyak quadrature involve weights which are negative ([sullivan2015]_ page 177).
The size of the experiment is only known after the nodes and weights have been computed,
that is, after the :meth:`generateWithWeights` method is called.

**Method**

Let :math:`\mathcal{X} \subset \mathbb{R}^{\inputDim}` be the integration domain 
and let :math:`\model : \mathcal{X} \rightarrow \mathbb{R}^{\outputDim}` be an integrable 
function. 
Let :math:`\inputProbabilityDensityFunction : \mathcal{X} \rightarrow \mathbb{R}` be a probability density function.
The Smolyak experiment produces an approximation of the integral:

.. math::

    \int_{\mathcal{X}} \model(\vect{x}) \inputProbabilityDensityFunction(\vect{x}) d\vect{x}
    \approx \sum_{i = 1}^{s_t} w_i \model\left(\vect{x}_i\right)

where :math:`s_t \in \mathbb{N}` is the size of the Smolyak
design of experiments, :math:`w_1, ..., w_{s_t} \in \mathbb{R}` are the
weights and :math:`\vect{x}_1, ..., \vect{x}_{s_t} \in \mathbb{R}^{\inputDim}`
are the nodes. 

Let :math:`\vect{k} = (k_1, ..., k_{\inputDim}) \in (\mathbb{N}^\star)^{\inputDim}` 
be the multi-index where

.. math::

    \mathbb{N}^\star = \{1, 2, ... \}

is the set of natural numbers without zero.
Consider the 1 and the infinity norms ([lemaitre2010]_ page 57, eq. 3.28):

.. math::

    \|\vect{k}\|_1 = \sum_{i = 1}^{\inputDim} k_i \qquad
    \|\vect{k}\|_\infty = \max_{i = 1, ..., \inputDim} k_i.

for any :math:`\vect{k} \in (\mathbb{N}^\star)^{\inputDim}`.

Let :math:`\ell` be an integer representing the level of the quadrature.
Let :math:`Q_{\ell}^{(1)}` be a marginal quadrature of level :math:`\ell`.
This marginal quadrature must have dimension 1 as is suggested by the exponent in the
notation :math:`Q_{\ell}^{(1)}`.
Depending on the level :math:`\ell`, we can compute the actual number of nodes
depending on a particular choice of that number of nodes and depending
on the quadrature rule.
The tensor product quadrature is:

.. math::

    T_\ell^{(\inputDim)} = Q_{\ell}^{(1)} \otimes \cdots \otimes Q_{\ell}^{(1)}.

In the previous equation, the marginal quadratures are not necessarily 
of the same type. 
For example, if the dimension is equal to 2, the first marginal quadrature may 
be a Gaussian quadrature while the second one may be a random experiment, such 
as a Monte-Carlo design of experiment.

Let :math:`Q_0^{(1)} = \emptyset` be the empty quadrature. 
For any integer :math:`k \in \mathbb{N}`, let :math:`\Delta_k^{(1)}` be the 
difference quadrature defined by:

.. math::

    \Delta_{k}^{(1)} = Q_{k}^{(1)} - Q_{k - 1}^{(1)}.    

Therefore, the quadrature formula :math:`Q_\ell` can be expressed depending
on difference quadratures:

.. math::

    Q_\ell^{(1)} = \sum_{k = 1}^\ell \Delta_k^{(1)}.

for any :math:`\ell \geq 1`.
The following equation provides an equivalent equation for the tensor product
quadrature ([lemaitre2010]_ page 57, eq. 3.30):

.. math::

    T_\ell^{(\inputDim)} = \sum_{\|\vect{k}\|_\infty \leq \ell}
    \Delta_{k_1}^{(1)} \otimes \cdots \otimes \Delta_{k_{\inputDim}}^{(1)}.

The significant part of the previous equation is the set of multi-indices
:math:`\|\vect{k}\|_\infty \leq \ell`, which may be very large
depending on the dimension of the problem.

One of the ways to reduce the size of this set is to consider the smaller set
of multi-indices such that :math:`\|\vect{k}\|_1 \leq \ell + {\inputDim} - 1`. The sparse
quadrature ([lemaitre2010]_ page 57, eq. 3.29, [gerstner1998]_  page 214)
is introduced in the following definition.

The Smolyak sparse quadrature formula at level :math:`\ell` is:

.. math::

    S_\ell^{(\inputDim)} = \sum_{\|\vect{k}\|_1 \leq \ell + \inputDim - 1} 
    \Delta_{k_1}^{(1)} \otimes \cdots \otimes \Delta_{k_{\inputDim}}^{(1)}

for any :math:`\ell \geq 1`. 

As shown by the previous equation, for a given multi-index :math:`\vect{k}`
the Smolyak quadrature requires to set the level of each marginal experiment to
an integer which depends on the multi-index.
This is done using the :meth:`setLevel` method of the marginal quadrature.

The following formula expresses the multivariate quadrature in terms of
combinations univariate quadratures, known as the *combination technique*.
This method combines elementary tensorized quadratures, summing them
depending on the binomial coefficient.
The sparse quadrature formula at level :math:`\ell` is:

.. math::

    S_\ell^{(\inputDim)} = \sum_{\ell \leq \|\vect{k}\|_1 \leq \ell + \inputDim - 1} 
    (-1)^{\ell + \inputDim - \|\vect{k}\|_1 - 1} 
    {\inputDim - 1 \choose \|\vect{k}\|_1 - \ell} 
    Q_{k_1}^{(1)} \otimes \cdots \otimes Q_{k_{\inputDim}}^{(1)}

for any :math:`\ell \geq 1` where the binomial coefficient is:

.. math::

    {n \choose m} = \frac{n!}{m! (n - m)!}

for any integers :math:`n \geq 0` and :math:`0 \leq m \leq n`.

**Merge duplicate nodes**

The Smolyak quadrature requires to merge the potentially duplicated nodes of the
elementary quadratures.
To do so, a dictionary is used so that unique nodes only are kept.
This algorithm is enabled by default, but it can be disabled with the
`SmolyakExperiment-MergeQuadrature` boolean key of the :class:`~openturns.ResourceMap`.
This can reduce the number of nodes, which may be particularly efficient when the
marginal quadrature rule is nested such as, for example, with Fejér or Clenshaw-Curtis
quadrature rule.

Assume we want to merge two candidate nodes :math:`\vect{x}_1, \vect{x}_2 \in \mathbb{R}^{\inputDim}`,
associated with the two weights :math:`w_1, w_2 \in \mathbb{R}`.
Let :math:`\vect{x}_1'` be the merged node and :math:`w_1'` the merged weight.
Then the merged node is

.. math::

    \vect{x}_1' = \vect{x}_1

and the merged weight is:


.. math::

    w_1' = w_1 + w_2.

If, however, the elementary quadrature rules have nodes which are computed up to some
rounding error, the merging algorithm may not detect that two nodes which are
close to each other are, indeed, the same.
This is why rounding errors must be taken into account.
The criterion is based on a mixed absolute and relative tolerance.
Let :math:`\epsilon_r, \epsilon_a > 0` be the relative and absolute tolerances.
These parameters can be set using the :class:`~openturns.ResourceMap` keys
`SmolyakExperiment-MergeRelativeEpsilon` and `SmolyakExperiment-MergeAbsoluteEpsilon`.
Two candidate nodes :math:`\vect{x}_1, \vect{x}_2 \in \mathbb{R}^{\inputDim}`
are close to each other and are to be merged if:

.. math::

    |(x_1)_i - (x_2)_i| \leq \epsilon_a + \epsilon_r
        \max\left(\left|(x_1)_i\right|, \left|(x_2)_i\right|\right).

for :math:`i = 1, ..., \inputDim`.

If the bounds of the input distribution are either very close to zero
or very large, then the default settings may not work properly: in this case,
please fine tune the parameters to match your needs.

**Polynomial exactness**

The Smolyak quadrature rule is exact for polynomials up to some degree.
More precisely, for any :math:`m_i \in \Nset`, let :math:`\mathcal{P}_{m_i}^{(1)}`
be the set of mono-variable polynomials of degree lower or equal to :math:`m_i`.
Consider the tensor product of 1D polynomials:

.. math::

    \bigotimes_{i = 1}^\inputDim \mathcal{P}_{m_i}^{(1)}
    = 
    \left\{
    (x_1, ..., x_\inputDim)\in\Rset^\inputDim
    \rightarrow \prod_{i = 1}^\inputDim p_i(x_i) \in \Rset, \quad 
    p_i \in \mathcal{P}_{m_i}^{(1)}
    \right\}.

Assume that :math:`m_{k_i}` is the polynomial degree of exactness of the :math:`i`-th 
marginal univariate quadrature rule using :math:`k_i` nodes, for any 
:math:`i \in \{1,...,\inputDim\}`.
Therefore the Smolyak quadrature is exact for all polynomials of the non-classical 
vector space (see [novak1999]_ theorem 2 page 88 and [peter2019]_ section 5.4 page 2-17):

.. math::

    \operatorname{Span}
    \left\{
    \mathcal{P}_{m_{k_1}}^{(1)} \otimes \cdots \otimes \mathcal{P}_{m_{k_\inputDim}}^{(1)}
    \quad / \quad \|\boldsymbol{k}\|_1 = \ell + \inputDim - 1
    \right\}.

**Accuracy**
    
The following equation presents the absolute error of a sparse
quadrature ([sullivan2015]_ page 177, eq. 9.10).
Assume that :math:`\model \in \mathcal{C}^r([0, 1]^{\inputDim})` and that we use a sparse
quadrature with :math:`n_\ell` nodes at level :math:`\ell`.
In this particular case, the probability density function :math:`\inputProbabilityDensityFunction` is
equal to 1. 
Therefore,

.. math::

    \left|\int_{[0, 1]^{\inputDim}} \model(\vect{x}) d\vect{x} 
    - S_\ell(\model)\right|
    = O \left(n_\ell^{-r} (\log(n_\ell))^{(\inputDim - 1)(r + 1)}\right).

See Also
--------
WeightedExperiment

Examples
--------
In the following example, we create Smolyak quadrature using two Gauss-Legendre
marginal quadratures.

>>> import openturns as ot
>>> experiment1 = ot.GaussProductExperiment(ot.Uniform(0.0, 1.0))
>>> experiment2 = ot.GaussProductExperiment(ot.Uniform(0.0, 1.0))
>>> collection = [experiment1, experiment2]
>>> level = 3
>>> multivariate_experiment = ot.SmolyakExperiment(collection, level)
>>> nodes, weights = multivariate_experiment.generateWithWeights()
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::SmolyakExperiment::getExperimentCollection
"Get the marginals of the experiment.

Returns
-------
experiments : list of :class:`~openturns.WeightedExperiment`
    List of the marginals of the experiment."

// ---------------------------------------------------------------------

%feature("docstring") OT::SmolyakExperiment::setExperimentCollection
"Set the marginals of the experiment.

Parameters
----------
experiments : list of :class:`~openturns.WeightedExperiment`
    List of the marginals of the experiment."

// ---------------------------------------------------------------------

%feature("docstring") OT::SmolyakExperiment::computeCombination
"Compute the indices involved in the quadrature.

Returns
-------
indicesCollection : list of :class:`~openturns.IndicesCollection`
    List of the multi-indices involved in Smolyak's quadrature."

// ---------------------------------------------------------------------

%feature("docstring") OT::SmolyakExperiment::getLevel
R"RAW(Get the level of the experiment.

Returns
-------
level : int
    Level value :math:`\ell \geq 1`.)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::SmolyakExperiment::setLevel
R"RAW(Set the level of the experiment.

Parameters
----------
level : int
    Level value :math:`\ell \geq 1`.)RAW"
